/*
 * EMU][ Apple ][-class emulator
 * Copyright (C) 2002- 2004 by the EMU][ Project/Dapple ][ Team
 *
 * Component:  CPU6502 - standard 6502 core
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, version 2.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 *  Current exception for ASMLIB.O linkage, if Z80.C is not used
 *
 * 20040428  New headers generated for v0.4 release
 *
 */


/*----------------------------------------


        class   cpu6502


        see cpu6502.h for include file


        (c) by Holger Picker 2004


----------------------------------------*/


#ifndef DEF_INC_CPU6502_C
#define DEF_INC_CPU6502_C


#include <stdio.h>
#include <stdlib.h>
#include "cpu6502.h"
#include "..\libs\general.h"    // found in cpu6502.h
#include "..\libs\raf.h"        // found in cpu6502.h


// processor cycle steps

enum {

        CPU_GURU,       //      undefined opcode

        CPU_OPC,        //      execute opcode



        CPU_CLC_2,
        CPU_SEC_2,
        CPU_CLI_2,
        CPU_SEI_2,
        CPU_CLD_2,
        CPU_SED_2,
        CPU_CLV_2,



        CPU_TAX_2,
        CPU_TAY_2,
        CPU_TXA_2,
        CPU_TYA_2,
        CPU_TSX_2,
        CPU_TXS_2,



        CPU_PHA_2,
        CPU_PHA_3,
        CPU_PHX_2,
        CPU_PHX_3,
        CPU_PHY_2,
        CPU_PHY_3,
        CPU_PHP_2,
        CPU_PHP_3,
        CPU_PLA_2,
        CPU_PLA_3,
        CPU_PLA_4,
        CPU_PLX_2,
        CPU_PLX_3,
        CPU_PLX_4,
        CPU_PLY_2,
        CPU_PLY_3,
        CPU_PLY_4,
        CPU_PLP_2,
        CPU_PLP_3,
        CPU_PLP_4,



        CPU_INA_2,
        CPU_INX_2,
        CPU_INY_2,
        CPU_DEA_2,
        CPU_DEX_2,
        CPU_DEY_2,



        CPU_NOP_2,




        CPU_LDAI_2,
        CPU_LDA,
        CPU_STA,
        CPU_LDXI_2,
        CPU_LDX,
        CPU_STX,
        CPU_LDYI_2,
        CPU_LDY,
        CPU_STY,



        CPU_CMPI_2,
        CPU_CMP,
        CPU_CPXI_2,
        CPU_CPX,
        CPU_CPYI_2,
        CPU_CPY,
        CPU_BIT,



        CPU_ANDI_2,
        CPU_AND,
        CPU_EORI_2,
        CPU_EOR,
        CPU_ORAI_2,
        CPU_ORA,



        CPU_ADCI_2,
        CPU_ADC,
        CPU_SBCI_2,
        CPU_SBC,



        CPU_ASL_2,
        CPU_ASL,
        CPU_LSR_2,
        CPU_LSR,
        CPU_ROL_2,
        CPU_ROL,
        CPU_ROR_2,
        CPU_ROR,



        CPU_DEC,
        CPU_INC,



        CPU_BPL_2,
        CPU_BMI_2,
        CPU_BNE_2,
        CPU_BEQ_2,
        CPU_BCC_2,
        CPU_BCS_2,
        CPU_BVC_2,
        CPU_BVS_2,
        CPU_BRA_2,
        CPU_BRA_3,
        CPU_BRAM_4,
        CPU_BRAP_4,



        CPU_JMPABS_2,
        CPU_JMPABS_3,
        CPU_JMPABS_4,



        CPU_JMPIABS_2,
        CPU_JMPIABS_3,
        CPU_JMPIABS_4,
        CPU_JMPIABS_5,
        CPU_JMPIABS_6,



        CPU_JSRABS_2,
        CPU_JSRABS_3,
        CPU_JSRABS_4,
        CPU_JSRABS_5,
        CPU_JSRABS_6,



        CPU_RTS_2,
        CPU_RTS_3,
        CPU_RTS_4,
        CPU_RTS_5,
        CPU_RTS_6,



        CPU_RTI_2,
        CPU_RTI_3,
        CPU_RTI_4,
        CPU_RTI_5,
        CPU_RTI_6,



        CPU_ZP_2,
        CPU_MZP_2,
        CPU_MZP_3,
        CPU_MZP_4,
        CPU_ZPX_2,
        CPU_ZPX_3,
        CPU_MZPX_2,
        CPU_MZPX_3,
        CPU_MZPX_4,
        CPU_MZPX_5,
        CPU_ZPY_2,
        CPU_ZPY_3,
        CPU_ABS_2,
        CPU_ABS_3,
        CPU_MABS_2,
        CPU_MABS_3,
        CPU_MABS_4,
        CPU_MABS_5,
        CPU_ABSX_2,
        CPU_ABSX_3,
        CPU_ABSX_4,
        CPU_MABSX_2,
        CPU_MABSX_3,
        CPU_MABSX_4,
        CPU_MABSX_5,
        CPU_MABSX_6,
        CPU_ABSY_2,
        CPU_ABSY_3,
        CPU_ABSY_4,
        CPU_IZP_2,
        CPU_IZP_3,
        CPU_IZP_4,
        CPU_IZPX_2,
        CPU_IZPX_3,
        CPU_IZPX_4,
        CPU_IZPX_5,
        CPU_IZPY_2,
        CPU_IZPY_3,
        CPU_IZPY_4,


        CPU_BRK_2,
        CPU_BRK_3,
        CPU_BRK_4,
        CPU_BRK_5,
        CPU_BRK_6,



        CPU_COP_2,



        CPU_RESET_2,
        CPU_RESET_3,
        CPU_RESET_4,
        CPU_RESET_5,
        CPU_RESET_6,
        CPU_RESET_7,



        CPU_IRQ_2,
        CPU_IRQ_3,
        CPU_IRQ_4,
        CPU_IRQ_5,
        CPU_IRQ_6,
        CPU_IRQ_7,


        CPU_NMI_2,
        CPU_NMI_3,
        CPU_NMI_4,
        CPU_NMI_5,
        CPU_NMI_6,
        CPU_NMI_7

};



/*----------------------------------------

 MODULE cpu6502init     (cpu6502 *cpuptr)
                        ()

        initialize cpu6502

----------------------------------------*/

      unsigned char *cpu6502init(cpu6502 *cpuptr, raf *rafptr) {

        cpuptr->rafptr          = rafptr;
        cpuptr->areg            = 0;
        cpuptr->xreg            = 0;
        cpuptr->yreg            = 0;
        cpuptr->pcreg           = 0;
        cpuptr->spreg           = 0xff;
        cpuptr->nflag           = 0;
        cpuptr->vflag           = 0;
        cpuptr->bflag           = 0;
        cpuptr->dflag           = 0;
        cpuptr->zflag           = 0;
        cpuptr->cflag           = 0;
        cpuptr->step            = CPU_OPC;
        cpuptr->cycle           = 0;
        cpuptr->opcode          = 0;
        cpuptr->addr            = 0;
        cpuptr->data            = 0;
        cpuptr->stateflags      = 0;
        cpuptr->cputype         = CPU_6502;
        cpuptr->breakpoint      = 0xffffffff;
        cpu6502reset(cpuptr);

        return NULL;

      } /* cpu6502init */


/*----------------------------------------

 MODULE cpu6502reset    (cpu6502 *cpuptr)
                        ()

        reset cpu6502

----------------------------------------*/

      unsigned char *cpu6502reset(cpu6502 *cpuptr) {

        cpuptr->stateflags = cpuptr->stateflags | CPU_STATERESET;
        return NULL;

      } /* cpu6502reset */


/*----------------------------------------

 MODULE cpu6502setstate (cpu6502 *cpuptr)
                        (card32 value)

        set statebit of cpu

----------------------------------------*/

      void cpu6502setstate(cpu6502 *cpuptr, card32 value) {

        value = value &  (CPU_STATEHALT | CPU_STATERESET | CPU_STATETRACE
                        | CPU_STATEIRQ1 | CPU_STATEIRQ2
                        | CPU_STATENMI1 | CPU_STATENMI2)
                      & (~cpuptr->stateflags);

        if (value & CPU_STATEHALT) {
          tasklogstring(LOG_VERBOSE, "CPU halted");
          tasklogreturn(LOG_VERBOSE);
//        cpulight(0);
//        screenupdate = 1;
        }
        if (value & CPU_STATERESET) {
          tasklogstring(LOG_VERBOSE, "CPU Reset triggered");
          tasklogreturn(LOG_VERBOSE);
        }
        if (value & CPU_STATETRACE) {
          tasklogstring(LOG_VERBOSE, "CPU trace mode set");
          tasklogreturn(LOG_VERBOSE);
        }
//      if (value & CPU_STATEIRQ1) {
//        tasklogstring(LOG_VERBOSE, "CPU IRQ1 triggered");
//        tasklogreturn(LOG_VERBOSE);
//      }
//      if (value & CPU_STATEIRQ2) {
//        tasklogstring(LOG_VERBOSE, "CPU IRQ2 triggered");
//        tasklogreturn(LOG_VERBOSE);
//      }
        if (value & CPU_STATENMI1) {
          tasklogstring(LOG_VERBOSE, "CPU NMI1 triggered");
          tasklogreturn(LOG_VERBOSE);
        }
//      if (value & CPU_STATENMI2) {
//        tasklogstring(LOG_VERBOSE, "CPU NMI2 triggered");
//        tasklogreturn(LOG_VERBOSE);
//      }

        cpuptr->stateflags = cpuptr->stateflags | value;

      } // cpu6502setstate


/*----------------------------------------

 MODULE cpu6502clearstate       (cpu6502 *cpuptr)
                                (card32 value)

        clear statebit of cpu

----------------------------------------*/


      void cpu6502clearstate(cpu6502 *cpuptr, card32 value) {

        value = value
                &  (CPU_STATEHALT | CPU_STATERESET | CPU_STATETRACE | CPU_STATEBPT  | CPU_STATEGURU
                  | CPU_STATEIRQ1 | CPU_STATEIRQ2
                  | CPU_STATENMI1 | CPU_STATENMI2)
                & (cpuptr->stateflags);


        if (value & CPU_STATERESET) {
          tasklogstring(LOG_VERBOSE, "CPU Reset cleared");
          tasklogreturn(LOG_VERBOSE);
        }
        if (value & CPU_STATEHALT) {
          tasklogstring(LOG_VERBOSE, "CPU started");
          tasklogreturn(LOG_VERBOSE);
//        cpulight(1);
//        screenupdate = 1;
        }
        if (value & CPU_STATEGURU) {    // if it was an illegal opcode ==> reset whole machine
//        c64reset();
        }
//      if (value & CPU_STATEIRQ1) {
//        tasklogstring(LOG_VERBOSE, "CPU IRQ1 cleared");
//        tasklogreturn(LOG_VERBOSE);
//      }
//      if (value & CPU_STATEIRQ2) {
//        tasklogstring(LOG_VERBOSE, "CPU IRQ2 cleared");
//        tasklogreturn(LOG_VERBOSE);
//      }
//      if (value & CPU_STATENMI1) {
//        tasklogstring(LOG_VERBOSE, "CPU NMI1 cleared");
//        tasklogreturn(LOG_VERBOSE);
//      }
//      if (value & CPU_STATENMI2) {
//        tasklogstring(LOG_VERBOSE, "CPU NMI2 cleared");
//        tasklogreturn(LOG_VERBOSE);
//      }

/* don't write to log because of too many trace exceptions */
/*      if (value & CPU_STATETRACE) {
          tasklogstring(LOG_VERBOSE, "CPU trace mode cleared");
          tasklogreturn(LOG_VERBOSE);
        }
*/
        cpuptr->stateflags = cpuptr->stateflags & (~(value));

      } // cpu6502clearstate


/*----------------------------------------

 unsigned int   cpu6502getstate(cpu6502 *cpuptr)

        get stateflags

----------------------------------------*/


      unsigned int cpu6502getstate(cpu6502 *cpuptr) {

        return cpuptr->stateflags;

      } // cpu6502getstate


/*----------------------------------------

 MODULE cpu6502fastsethalt      (cpu6502 *cpuptr)
                                ()

        set halt

----------------------------------------*/


      void cpu6502fastsethalt(cpu6502 *cpuptr) {

        cpuptr->stateflags = cpuptr->stateflags | CPU_STATEHALT;
//      cpulight(0);

      } /* cpu6502fastsethalt */


/*----------------------------------------

 MODULE cpu6502fastclearhalt    (cpu6502 *cpuptr)
                                ()

        clear halt

----------------------------------------*/


      void cpu6502fastclearhalt(cpu6502 *cpuptr) {

        cpuptr->stateflags = cpuptr->stateflags & ~(CPU_STATEHALT);
//      cpulight(1);

      } // cpu6502fastclearhalt


/*----------------------------------------

 void   cpu6502flagslog (cpu6502 *cpuptr, unsigned int mode)

        write flags to log

----------------------------------------*/

      void cpu6502flagslog(cpu6502 *cpuptr, unsigned int mode) {
        register unsigned char *stringptr;
        unsigned char flagstring[10];

        tasklogstring(mode, "flags: ");
        stringptr = &flagstring[0];
        if (cpuptr->nflag)      { *stringptr++ = 'N'; }
        else                    { *stringptr++ = '-'; }
        if (cpuptr->vflag)      { *stringptr++ = 'V'; }
        else                    { *stringptr++ = '-'; }
        *stringptr++    = '.';
        if (cpuptr->bflag)      { *stringptr++ = 'B'; }
        else                    { *stringptr++ = '-'; }
        if (cpuptr->dflag)      { *stringptr++ = 'D'; }
        else                    { *stringptr++ = '-'; }
        if (cpuptr->iflag)      { *stringptr++ = 'I'; }
        else                    { *stringptr++ = '-'; }
        if (cpuptr->zflag)      { *stringptr++ = 'Z'; }
        else                    { *stringptr++ = '-'; }
        if (cpuptr->cflag)      { *stringptr++ = 'C'; }
        else                    { *stringptr++ = '-'; }
        *stringptr      = '\0';
        tasklogstring(mode, &flagstring[0]);

      } // cpu6502setflagslog


#define defsetflags(value) \
        cpuptr->nflag = (value & 0x80)  ? 1 : 0; \
        cpuptr->zflag = (value)         ? 0 : 1;


/*----------------------------------------

 MODULE cpu6502setflagsnz(cpu6502 *cpuptr, card8 value)
                        ()

        set N and Z according to value

----------------------------------------*/

      void cpu6502setflagsnz(cpu6502 *cpuptr, card8 value) {

        cpuptr->nflag = (value & 0x80)  ? 1 : 0;
        cpuptr->zflag = (value)         ? 0 : 1;

      } // cpu6502setflagsnz


      card8 cpu6502getflags(cpu6502 *cpuptr) {

        return
            ((cpuptr->nflag) ? 0x80 : 0)
          | ((cpuptr->vflag) ? 0x40 : 0)
          | 0x20
//        | ((cpuptr->bflag) ? 0x10 : 0)
          | ((cpuptr->dflag) ? 0x08 : 0)
          | ((cpuptr->iflag) ? 0x04 : 0)
          | ((cpuptr->zflag) ? 0x02 : 0)
          | ((cpuptr->cflag) ? 0x01 : 0);

      } // cpu6502getflags


      void cpu6502setflags(cpu6502 *cpuptr, unsigned int value) {

        cpuptr->nflag = (value & 0x80) ? 1 : 0;
        cpuptr->vflag = (value & 0x40) ? 1 : 0;
//      cpuptr->bflag = (value & 0x10) ? 1 : 0;
        cpuptr->dflag = (value & 0x08) ? 1 : 0;
        cpuptr->iflag = (value & 0x04) ? 1 : 0;
        cpuptr->zflag = (value & 0x02) ? 1 : 0;
        cpuptr->cflag = (value & 0x01) ? 1 : 0;

      } // cpu6502setflags


// Adc

/* The following code was provided by Mr. Scott Hemphill. */
/* Thanks a lot! */

      void cpu6502adc(cpu6502 *cpuptr, unsigned int value) {
        register unsigned int w;

        if ((cpuptr->areg ^ value) & 0x80) {
          cpuptr->vflag = 0;
        }
        else {
          cpuptr->vflag = 1;
        }

        if (cpuptr->dflag) {
          w = (cpuptr->areg & 0xf) + (value & 0xf) + cpuptr->cflag;
          if (w >= 10) {
            w = 0x10 | ((w+6)&0xf);
          }
          w += (cpuptr->areg & 0xf0) + (value & 0xf0);
          if (w >= 160) {
            cpuptr->cflag = 1;
            if (cpuptr->vflag && w >= 0x180) cpuptr->vflag = 0;
            w += 0x60;
          }
          else {
            cpuptr->cflag = 0;
            if (cpuptr->vflag && w < 0x80) cpuptr->vflag = 0;
          }
        }
        else {
          w = cpuptr->areg + value + cpuptr->cflag;
          if (w >= 0x100) {
            cpuptr->cflag = 1;
            if (cpuptr->vflag && (w >= 0x180)) cpuptr->vflag = 0;
          }
          else {
            cpuptr->cflag = 0;
            if (cpuptr->vflag && (w  < 0x80)) cpuptr->vflag = 0;
          }
        }
        cpuptr->areg  = (card8)w;
        cpuptr->nflag = (cpuptr->areg >= 0x80) ? 1 : 0;
        cpuptr->zflag = (cpuptr->areg == 0)    ? 1 : 0;
      } /* adc */


/* Sbc */

/* The following code was provided by Mr. Scott Hemphill. */
/* Thanks a lot again! */

      void cpu6502sbc(cpu6502 *cpuptr, unsigned int value) {
        register unsigned int w;
        register unsigned char temp;

        if ((cpuptr->areg ^ value) & 0x80) {
          cpuptr->vflag = 1;
        }
        else {
          cpuptr->vflag = 0;
        }

        if (cpuptr->dflag) {            /* decimal subtraction */
          temp = 0xf + (cpuptr->areg & 0xf) - (value & 0xf) + (cpuptr->cflag);
          if (temp < 0x10) {
            w = 0;
            temp -= 6;
          }
          else {
            w = 0x10;
            temp -= 0x10;
          }
          w += 0xf0 + (cpuptr->areg & 0xf0) - (value & 0xf0);
          if (w < 0x100) {
            cpuptr->cflag = 0;
            if (cpuptr->vflag && w < 0x80) cpuptr->vflag = 0;
            w -= 0x60;
          }
          else {
            cpuptr->cflag = 1;
            if ((cpuptr->vflag) && w >= 0x180) cpuptr->vflag = 0;
          }
          w += temp;
        }
        else {                  /* standard binary subtraction */
          w = 0xff + cpuptr->areg - value + cpuptr->cflag;
          if (w < 0x100) {
            cpuptr->cflag = 0;
            if (cpuptr->vflag && (w < 0x80)) cpuptr->vflag = 0;
          }
          else {
            cpuptr->cflag = 1;
            if (cpuptr->vflag && (w >= 0x180)) cpuptr->vflag = 0;
          }
        }
        cpuptr->areg  = (card8)w;
        cpuptr->nflag = (cpuptr->areg >= 0x80) ? 1 : 0;
        cpuptr->zflag = (cpuptr->areg == 0)    ? 1 : 0;
      } /* sbc */


/*----------------------------------------

 MODULE cpu6502step     (cpu6502 *cpuptr)
                        ()

        step cpu6502

----------------------------------------*/

static card32 cpu6502opcstep[256] = {

        CPU_GURU,               // 0x00
        CPU_IZPX_2,             // 0x01         ORA (zp,x)
        CPU_COP_2,              // 0x02         coprocessor instruction
        CPU_GURU,               // 0x03
        CPU_GURU,               // 0x04
        CPU_ZP_2,               // 0x05         ORA zp
        CPU_MZP_2,              // 0x06         ASL zp
        CPU_GURU,               // 0x07
        CPU_PHP_2,              // 0x08         PHP
        CPU_ORAI_2,             // 0x09         ORA #
        CPU_ASL_2,              // 0x0a         ASL
        CPU_GURU,               // 0x0b
        CPU_GURU,               // 0x0c
        CPU_ABS_2,              // 0x0d         ORA abs
        CPU_MABS_2,             // 0x0e         ASL abs
        CPU_GURU,               // 0x0f

        CPU_BPL_2,              // 0x10         BPL
        CPU_IZPY_2,             // 0x11         ORA (zp),y
        CPU_GURU,               // 0x12
        CPU_GURU,               // 0x13
        CPU_GURU,               // 0x14
        CPU_ZPX_2,              // 0x15         ORA zp,x
        CPU_MZPX_2,             // 0x16         ASL zp,x
        CPU_GURU,               // 0x17
        CPU_CLC_2,              // 0x18         CLC
        CPU_ABSY_2,             // 0x19         ORA abs,y
        CPU_GURU,               // 0x1a
        CPU_GURU,               // 0x1b
        CPU_GURU,               // 0x1c
        CPU_ABSX_2,             // 0x1d         ORA abs,x
        CPU_MABSX_2,            // 0x1e         ASL abs,x
        CPU_GURU,               // 0x1f

        CPU_JSRABS_2,           // 0x20         JSR abs
        CPU_IZPX_2,             // 0x21         AND (zp,x)
        CPU_GURU,               // 0x22
        CPU_GURU,               // 0x23
        CPU_ZP_2,               // 0x24         BIT zp
        CPU_ZP_2,               // 0x25         AND zp
        CPU_MZP_2,              // 0x26         ROL zp
        CPU_GURU,               // 0x27
        CPU_PLP_2,              // 0x28         PLP
        CPU_ANDI_2,             // 0x29         AND #
        CPU_ROL_2,              // 0x2a         ROL
        CPU_GURU,               // 0x2b
        CPU_ABS_2,              // 0x2c         BIT abs
        CPU_ABS_2,              // 0x2d         AND abs
        CPU_MABS_2,             // 0x2e         ROL abs
        CPU_GURU,               // 0x2f

        CPU_BMI_2,              // 0x30         BMI
        CPU_IZPY_2,             // 0x31         AND (zp),y
        CPU_GURU,               // 0x32
        CPU_GURU,               // 0x33
        CPU_GURU,               // 0x34
        CPU_ZPX_2,              // 0x35         AND zp,x
        CPU_MZPX_2,             // 0x36         ROL zp,x
        CPU_GURU,               // 0x37
        CPU_SEC_2,              // 0x38         SEC
        CPU_ABSY_2,             // 0x39         AND abs,y
        CPU_GURU,               // 0x3a
        CPU_GURU,               // 0x3b
        CPU_GURU,               // 0x3c
        CPU_ABSX_2,             // 0x3d         AND abs,x
        CPU_MABSX_2,            // 0x3e         ROL abs,x
        CPU_GURU,               // 0x3f

        CPU_RTI_2,              // 0x40         RTI
        CPU_IZPX_2,             // 0x41         EOR (zp,x)
        CPU_GURU,               // 0x42
        CPU_GURU,               // 0x43
        CPU_GURU,               // 0x44
        CPU_ZP_2,               // 0x45         EOR zp
        CPU_MZP_2,              // 0x46         LSR zp
        CPU_GURU,               // 0x47
        CPU_PHA_2,              // 0x48         PHA
        CPU_EORI_2,             // 0x49         EOR #
        CPU_LSR_2,              // 0x4a         LSR
        CPU_GURU,               // 0x4b
        CPU_JMPABS_2,           // 0x4c         JMP abs
        CPU_ABS_2,              // 0x4d         EOR abs
        CPU_MABS_2,             // 0x4e         LSR abs
        CPU_GURU,               // 0x4f

        CPU_BVC_2,              // 0x50         BVC
        CPU_IZPY_2,             // 0x51         EOR (zp),y
        CPU_GURU,               // 0x52
        CPU_GURU,               // 0x53
        CPU_GURU,               // 0x54
        CPU_ZPX_2,              // 0x55         EOR zp,x
        CPU_MZPX_2,             // 0x56         LSR zp,x
        CPU_GURU,               // 0x57
        CPU_CLI_2,              // 0x58         CLI
        CPU_ABSY_2,             // 0x59         EOR abs,y
        CPU_GURU,               // 0x5a
        CPU_GURU,               // 0x5b
        CPU_GURU,               // 0x5c
        CPU_ABSX_2,             // 0x5d         EOR abs,x
        CPU_MABSX_2,            // 0x5e         LSR abs,x
        CPU_GURU,               // 0x5f

        CPU_RTS_2,              // 0x60         RTS
        CPU_IZPX_2,             // 0x61         ADC (zp,x)
        CPU_GURU,               // 0x62
        CPU_GURU,               // 0x63
        CPU_GURU,               // 0x64
        CPU_ZP_2,               // 0x65         ADC zp
        CPU_MZP_2,              // 0x66         ROR zp
        CPU_GURU,               // 0x67
        CPU_PLA_2,              // 0x68         PLA
        CPU_ADCI_2,             // 0x69         ADC #
        CPU_ROR_2,              // 0x6a         ROR
        CPU_GURU,               // 0x6b
        CPU_JMPIABS_2,          // 0x6c         JMP (abs)
        CPU_ABS_2,              // 0x6d         ADC abs
        CPU_MABS_2,             // 0x6e         ROR abs
        CPU_GURU,               // 0x6f

        CPU_BVS_2,              // 0x70         BVS
        CPU_IZPY_2,             // 0x71         ADC (zp),y
        CPU_GURU,               // 0x72
        CPU_GURU,               // 0x73
        CPU_GURU,               // 0x74
        CPU_ZPX_2,              // 0x75         ADC zp,x
        CPU_MZPX_2,             // 0x76         ROR zp,x
        CPU_GURU,               // 0x77
        CPU_SEI_2,              // 0x78         SEI
        CPU_ABSY_2,             // 0x79         ADC abs,y
        CPU_GURU,               // 0x7a
        CPU_GURU,               // 0x7b
        CPU_GURU,               // 0x7c
        CPU_ABSX_2,             // 0x7d         ADC abs
        CPU_MABSX_2,            // 0x7e         ROR abs,x
        CPU_GURU,               // 0x7f

        CPU_GURU,               // 0x80
        CPU_IZPX_2,             // 0x81         STA (zp,x)
        CPU_GURU,               // 0x82
        CPU_GURU,               // 0x83
        CPU_ZP_2,               // 0x84         STY zp
        CPU_ZP_2,               // 0x85         STA zp
        CPU_ZP_2,               // 0x86         STX zp
        CPU_GURU,               // 0x87
        CPU_DEY_2,              // 0x88         DEY
        CPU_GURU,               // 0x89
        CPU_TXA_2,              // 0x8a         TXA
        CPU_GURU,               // 0x8b
        CPU_ABS_2,              // 0x8c         STY abs
        CPU_ABS_2,              // 0x8d         STA abs
        CPU_ABS_2,              // 0x8e         STX abs
        CPU_GURU,               // 0x8f

        CPU_BCC_2,              // 0x90         BCC
        CPU_IZPY_2,             // 0x91         STA (zp),y
        CPU_GURU,               // 0x92
        CPU_GURU,               // 0x93
        CPU_ZPX_2,              // 0x94         STY zp,x
        CPU_ZPX_2,              // 0x95         STA zp,x
        CPU_ZPY_2,              // 0x96         STX zp,y
        CPU_GURU,               // 0x97
        CPU_TYA_2,              // 0x98         TYA
        CPU_ABSY_2,             // 0x99         STA abs,y
        CPU_TXS_2,              // 0x9a         TXS
        CPU_GURU,               // 0x9b
        CPU_GURU,               // 0x9c
        CPU_ABSX_2,             // 0x9d         STA abs,x
        CPU_GURU,               // 0x9e
        CPU_GURU,               // 0x9f

        CPU_LDYI_2,             // 0xa0         LDY #
        CPU_IZPX_2,             // 0xa1         LDA (zp,x)
        CPU_LDXI_2,             // 0xa2         LDX #
        CPU_GURU,               // 0xa3
        CPU_ZP_2,               // 0xa4         LDY zp
        CPU_ZP_2,               // 0xa5         LDA zp
        CPU_ZP_2,               // 0xa6         LDX zp
        CPU_GURU,               // 0xa7
        CPU_TAY_2,              // 0xa8         TAY
        CPU_LDAI_2,             // 0xa9         LDA #
        CPU_TAX_2,              // 0xaa         TAX
        CPU_GURU,               // 0xab
        CPU_ABS_2,              // 0xac         LDY abs
        CPU_ABS_2,              // 0xad         LDA abs
        CPU_ABS_2,              // 0xae         LDX abs
        CPU_GURU,               // 0xaf

        CPU_BCS_2,              // 0xb0         BCS
        CPU_IZPY_2,             // 0xb1         LDA (zp),y
        CPU_GURU,               // 0xb2
        CPU_GURU,               // 0xb3
        CPU_ZPX_2,              // 0xb4         LDY zp,x
        CPU_ZPX_2,              // 0xb5         LDA zp,x
        CPU_ZPY_2,              // 0xb6         LDX zp,y
        CPU_GURU,               // 0xb7
        CPU_CLV_2,              // 0xb8         CLV
        CPU_ABSY_2,             // 0xb9         LDA abs,y
        CPU_TSX_2,              // 0xba         TSX
        CPU_GURU,               // 0xbb
        CPU_ABSX_2,             // 0xbc         LDY abs,x
        CPU_ABSX_2,             // 0xbd         LDA abs,x
        CPU_ABSY_2,             // 0xbe         LDX abs,y
        CPU_GURU,               // 0xbf

        CPU_CPYI_2,             // 0xc0         CPY #
        CPU_IZPX_2,             // 0xc1         CMP (zp,x)
        CPU_GURU,               // 0xc2
        CPU_GURU,               // 0xc3
        CPU_ZP_2,               // 0xc4         CPY zp
        CPU_ZP_2,               // 0xc5         CMP zp
        CPU_MZP_2,              // 0xc6         DEC zp
        CPU_GURU,               // 0xc7
        CPU_INY_2,              // 0xc8         INY
        CPU_CMPI_2,             // 0xc9         CMP #
        CPU_DEX_2,              // 0xca         DEX
        CPU_GURU,               // 0xcd
        CPU_ABS_2,              // 0xcc         CPY abs
        CPU_ABS_2,              // 0xcd         CMP abs
        CPU_MABS_2,             // 0xce         DEC abs
        CPU_GURU,               // 0xcf

        CPU_BNE_2,              // 0xd0         BNE
        CPU_IZPY_2,             // 0xd1         CMP (zp),y
        CPU_GURU,               // 0xd2
        CPU_GURU,               // 0xd3
        CPU_GURU,               // 0xd4
        CPU_ZPX_2,              // 0xd5         CMP zp,x
        CPU_MZPX_2,             // 0xd6         DEC zp,x
        CPU_GURU,               // 0xd7
        CPU_CLD_2,              // 0xd8         CLD
        CPU_ABSY_2,             // 0xd9         CMP abs,y
        CPU_GURU,               // 0xda
        CPU_GURU,               // 0xdb
        CPU_GURU,               // 0xdc
        CPU_ABSX_2,             // 0xdd         CMP abs,x
        CPU_MABSX_2,            // 0xde         DEC abs,x
        CPU_GURU,               // 0xdf

        CPU_CPXI_2,             // 0xe0         CPX #
        CPU_IZPX_2,             // 0xe1         SBC (zp,x)
        CPU_GURU,               // 0xe2
        CPU_GURU,               // 0xe3
        CPU_ZP_2,               // 0xe4         CPX zp
        CPU_ZP_2,               // 0xe5         SBC zp
        CPU_MZP_2,              // 0xe6         INC zp
        CPU_GURU,               // 0xe7
        CPU_INX_2,              // 0xe8         INX
        CPU_SBCI_2,             // 0xe9         SBC #
        CPU_NOP_2,              // 0xea         NOP
        CPU_GURU,               // 0xeb
        CPU_ABS_2,              // 0xec         CPX abs
        CPU_ABS_2,              // 0xed         SBC abs
        CPU_MABS_2,             // 0xee         INC abs
        CPU_GURU,               // 0xef

        CPU_BEQ_2,              // 0xf0         BEQ
        CPU_IZPY_2,             // 0xf1         SBC (zp),y
        CPU_GURU,               // 0xf2
        CPU_GURU,               // 0xf3
        CPU_GURU,               // 0xf4
        CPU_ZPX_2,              // 0xf5         SBC zp,x
        CPU_MZPX_2,             // 0xf6         INC zp,x
        CPU_GURU,               // 0xf7
        CPU_SED_2,              // 0xf8         SED
        CPU_ABSY_2,             // 0xf9         SBC abs,y
        CPU_GURU,               // 0xfa
        CPU_GURU,               // 0xfb
        CPU_GURU,               // 0xfc
        CPU_ABSX_2,             // 0xfd         SBC abs,x
        CPU_MABSX_2,            // 0xfe         INC abs,x
        CPU_GURU                // 0xff

        };


static card32 cpu6502opcstep2[256] = {

        CPU_GURU,               // 0x00
        CPU_ORA,                // 0x01         ORA (zp,x)
        CPU_GURU,               // 0x02
        CPU_GURU,               // 0x03
        CPU_GURU,               // 0x04
        CPU_ORA,                // 0x05         ORA zp
        CPU_ASL,                // 0x06         ASL zp
        CPU_GURU,               // 0x07
        CPU_GURU,               // 0x08         PHP
        CPU_GURU,               // 0x09         ORA #
        CPU_GURU,               // 0x0a         ASL
        CPU_GURU,               // 0x0b
        CPU_GURU,               // 0x0c
        CPU_ORA,                // 0x0d         ORA abs
        CPU_ASL,                // 0x0e         ASL abs
        CPU_GURU,               // 0x0f

        CPU_GURU,               // 0x10         BPL
        CPU_ORA,                // 0x11         ORA (zp),y
        CPU_GURU,               // 0x12
        CPU_GURU,               // 0x13
        CPU_GURU,               // 0x14
        CPU_ORA,                // 0x15         ORA zp,x
        CPU_ASL,                // 0x16         ASL zp,x
        CPU_GURU,               // 0x17
        CPU_GURU,               // 0x18         CLC
        CPU_ORA,                // 0x19         ORA abs,y
        CPU_GURU,               // 0x1a
        CPU_GURU,               // 0x1b
        CPU_GURU,               // 0x1c
        CPU_ORA,                // 0x1d         ORA abs,x
        CPU_ASL,                // 0x1e         ASL abs,x
        CPU_GURU,               // 0x1f

        CPU_GURU,               // 0x20         JSR abs
        CPU_AND,                // 0x21         AND (zp,x)
        CPU_GURU,               // 0x22
        CPU_GURU,               // 0x23
        CPU_BIT,                // 0x24         BIT
        CPU_AND,                // 0x25         AND zp
        CPU_ROL,                // 0x26         ROL zp
        CPU_GURU,               // 0x27
        CPU_GURU,               // 0x28         PLP
        CPU_GURU,               // 0x29         AND #
        CPU_GURU,               // 0x2a         ROL
        CPU_GURU,               // 0x2b
        CPU_BIT,                // 0x2c         BIT abs
        CPU_AND,                // 0x2d         AND abs
        CPU_ROL,                // 0x2e         ROL abs
        CPU_GURU,               // 0x2f

        CPU_GURU,               // 0x30         BMI
        CPU_AND,                // 0x31         AND (zp),y
        CPU_GURU,               // 0x32
        CPU_GURU,               // 0x33
        CPU_GURU,               // 0x34
        CPU_AND,                // 0x35         AND zp,x
        CPU_ROL,                // 0x36         ROL zp
        CPU_GURU,               // 0x37
        CPU_GURU,               // 0x38         SEC
        CPU_AND,                // 0x39         AND abs,y
        CPU_GURU,               // 0x3a
        CPU_GURU,               // 0x3b
        CPU_GURU,               // 0x3c
        CPU_AND,                // 0x3d         AND abs,x
        CPU_ROL,                // 0x3e         ROL abs,x
        CPU_GURU,               // 0x3f

        CPU_GURU,               // 0x40         RTI
        CPU_EOR,                // 0x41         EOR (zp,x)
        CPU_GURU,               // 0x42
        CPU_GURU,               // 0x43
        CPU_GURU,               // 0x44
        CPU_EOR,                // 0x45         EOR zp
        CPU_LSR,                // 0x46         LSR zp
        CPU_GURU,               // 0x47
        CPU_GURU,               // 0x48         PHA
        CPU_GURU,               // 0x49         EOR #
        CPU_GURU,               // 0x4a         LSR
        CPU_GURU,               // 0x4b
        CPU_GURU,               // 0x4c         JMP abs
        CPU_EOR,                // 0x4d         EOR abs
        CPU_LSR,                // 0x4e         LSR abs
        CPU_GURU,               // 0x4f

        CPU_GURU,               // 0x50         BVC
        CPU_EOR,                // 0x51         EOR (zp),y
        CPU_GURU,               // 0x52
        CPU_GURU,               // 0x53
        CPU_GURU,               // 0x54
        CPU_EOR,                // 0x55         EOR zp,x
        CPU_LSR,                // 0x56         LSR zp,x
        CPU_GURU,               // 0x57
        CPU_GURU,               // 0x58         CLI
        CPU_EOR,                // 0x59         EOR abs,y
        CPU_GURU,               // 0x5a
        CPU_GURU,               // 0x5b
        CPU_GURU,               // 0x5c
        CPU_EOR,                // 0x5d         EOR abs,x
        CPU_LSR,                // 0x5e         LSR abs,x
        CPU_GURU,               // 0x5f

        CPU_GURU,               // 0x60         RTS
        CPU_ADC,                // 0x61         ADC (zp,x)
        CPU_GURU,               // 0x62
        CPU_GURU,               // 0x63
        CPU_GURU,               // 0x64
        CPU_ADC,                // 0x65         ADC zp
        CPU_ROR,                // 0x66         ROR zp
        CPU_GURU,               // 0x67
        CPU_GURU,               // 0x68         PLA
        CPU_GURU,               // 0x69         ADC #
        CPU_GURU,               // 0x6a         ROR
        CPU_GURU,               // 0x6b
        CPU_GURU,               // 0x6c         JMP (abs)
        CPU_ADC,                // 0x6d         ADC abs
        CPU_ROR,                // 0x6e         ROR abs
        CPU_GURU,               // 0x6f

        CPU_GURU,               // 0x70         BVS
        CPU_ADC,                // 0x71         ADC (zp),y
        CPU_GURU,               // 0x72
        CPU_GURU,               // 0x73
        CPU_GURU,               // 0x74
        CPU_ADC,                // 0x75         ADC zp,x
        CPU_ROR,                // 0x76         ROR zp,x
        CPU_GURU,               // 0x77
        CPU_GURU,               // 0x78         SEI
        CPU_ADC,                // 0x79         ADC abs,y
        CPU_GURU,               // 0x7a
        CPU_GURU,               // 0x7b
        CPU_GURU,               // 0x7c
        CPU_ADC,                // 0x7d         ADC abs,x
        CPU_ROR,                // 0x7e         ROR abs,x
        CPU_GURU,               // 0x7f

        CPU_GURU,               // 0x80
        CPU_STA,                // 0x81         STA (zp,x)
        CPU_GURU,               // 0x82
        CPU_GURU,               // 0x83
        CPU_STY,                // 0x84         STY zp
        CPU_STA,                // 0x85         STA zp
        CPU_STX,                // 0x86         STX zp
        CPU_GURU,               // 0x87
        CPU_GURU,               // 0x88         DEY
        CPU_GURU,               // 0x89
        CPU_GURU,               // 0x8a         TXA
        CPU_GURU,               // 0x8b
        CPU_STY,                // 0x8c         STY abs
        CPU_STA,                // 0x8d         STA abs
        CPU_STX,                // 0x8e         STX abs
        CPU_GURU,               // 0x8f

        CPU_GURU,               // 0x90         BCC
        CPU_STA,                // 0x91         STA (zp),y
        CPU_GURU,               // 0x92
        CPU_GURU,               // 0x93
        CPU_STY,                // 0x94         STY zp,x
        CPU_STA,                // 0x95         STA zp,x
        CPU_STX,                // 0x96         STX zp,y
        CPU_GURU,               // 0x97
        CPU_GURU,               // 0x98         TYA
        CPU_STA,                // 0x99         STA abs,y
        CPU_GURU,               // 0x9a         TXS
        CPU_GURU,               // 0x9b
        CPU_GURU,               // 0x9c
        CPU_STA,                // 0x9d         STA abs,x
        CPU_GURU,               // 0x9e
        CPU_GURU,               // 0x9f

        CPU_GURU,               // 0xa0         LDY #
        CPU_LDA,                // 0xa1         LDA (zp,x)
        CPU_GURU,               // 0xa2         LDX #
        CPU_GURU,               // 0xa3
        CPU_LDY,                // 0xa4         LDY zp
        CPU_LDA,                // 0xa5         LDA zp
        CPU_LDX,                // 0xa6         LDX zp
        CPU_GURU,               // 0xa7
        CPU_GURU,               // 0xa8         TAY
        CPU_GURU,               // 0xa9         LDA #
        CPU_GURU,               // 0xaa         TAX
        CPU_GURU,               // 0xab
        CPU_LDY,                // 0xac         LDY abs
        CPU_LDA,                // 0xad         LDA abs
        CPU_LDX,                // 0xae         LDX abs
        CPU_GURU,               // 0xaf

        CPU_GURU,               // 0xb0         BCS
        CPU_LDA,                // 0xb1         LDA (zp),y
        CPU_GURU,               // 0xb2
        CPU_GURU,               // 0xb3
        CPU_LDY,                // 0xb4         LDY zp,x
        CPU_LDA,                // 0xb5         LDA zp,x
        CPU_LDX,                // 0xb6         LDX zp,y
        CPU_GURU,               // 0xb7
        CPU_GURU,               // 0xb8         CLV
        CPU_LDA,                // 0xb9         LDA abs,y
        CPU_GURU,               // 0xba         TSX
        CPU_GURU,               // 0xbb
        CPU_LDY,                // 0xbc         LDY abs,x
        CPU_LDA,                // 0xbd         LDA abs,x
        CPU_LDX,                // 0xbe         LDX abs,y
        CPU_GURU,               // 0xbf

        CPU_GURU,               // 0xc0         CPY #
        CPU_CMP,                // 0xc1         CMP (zp,x)
        CPU_GURU,               // 0xc2
        CPU_GURU,               // 0xc3
        CPU_CPY,                // 0xc4         CPY zp
        CPU_CMP,                // 0xc5         CMP zp
        CPU_DEC,                // 0xc6         DEC zp
        CPU_GURU,               // 0xc7
        CPU_GURU,               // 0xc8         INY
        CPU_GURU,               // 0xc9         CMP #
        CPU_GURU,               // 0xca         DEX
        CPU_GURU,               // 0xcb
        CPU_CPY,                // 0xcc         CPY abs
        CPU_CMP,                // 0xcd         CMP abs
        CPU_DEC,                // 0xce         DEC abs
        CPU_GURU,               // 0xcf

        CPU_GURU,               // 0xd0         BNE
        CPU_CMP,                // 0xd1         CMP (zp),y
        CPU_GURU,               // 0xd2
        CPU_GURU,               // 0xd3
        CPU_GURU,               // 0xd4
        CPU_CMP,                // 0xd5         CMP zp,x
        CPU_DEC,                // 0xd6         DEC zp,x
        CPU_GURU,               // 0xd7
        CPU_GURU,               // 0xd8         CLD
        CPU_CMP,                // 0xd9         CMP abs,y
        CPU_GURU,               // 0xda
        CPU_GURU,               // 0xdb
        CPU_GURU,               // 0xdc
        CPU_CMP,                // 0xdd         CMP abs,x
        CPU_DEC,                // 0xde         DEC abs,x
        CPU_GURU,               // 0xdf

        CPU_GURU,               // 0xe0         CPX #
        CPU_SBC,                // 0xe1         SBC (zp,x)
        CPU_GURU,               // 0xe2
        CPU_GURU,               // 0xe3
        CPU_CPX,                // 0xe4         CPX zp
        CPU_SBC,                // 0xe5         SBC zp
        CPU_INC,                // 0xe6         INC zp
        CPU_GURU,               // 0xe7
        CPU_GURU,               // 0xe8         INX
        CPU_GURU,               // 0xe9         SBC #
        CPU_GURU,               // 0xea         NOP
        CPU_GURU,               // 0xeb
        CPU_CPX,                // 0xec         CPX abs
        CPU_SBC,                // 0xed         SBC abs
        CPU_INC,                // 0xee         INC abs
        CPU_GURU,               // 0xef

        CPU_GURU,               // 0xf0         BEQ
        CPU_SBC,                // 0xf1         SBC (zp),y
        CPU_GURU,               // 0xf2
        CPU_GURU,               // 0xf3
        CPU_GURU,               // 0xf4
        CPU_SBC,                // 0xf5         SBC zp,x
        CPU_INC,                // 0xf6         INC zp,x
        CPU_GURU,               // 0xf7
        CPU_GURU,               // 0xf8         SED
        CPU_SBC,                // 0xf9         SBC abs,y
        CPU_GURU,               // 0xfa
        CPU_GURU,               // 0xfb
        CPU_GURU,               // 0xfc
        CPU_SBC,                // 0xfd         SBC abs,x
        CPU_INC,                // 0xfe         INC abs,x
        CPU_GURU                // 0xff

        };


      unsigned char *cpu6502step(cpu6502 *cpuptr) {
        static card8 data;

        switch (cpuptr->step) {

          case CPU_GURU :
                cpuptr->step            = CPU_OPC;
                cpuptr->stateflags      = cpuptr->stateflags | CPU_STATEGURU | CPU_STATEHALT;
                taskloghex32(LOG_VERBOSE, cpuptr->pcreg - 1);
                tasklogstring(LOG_VERBOSE, ": Guru error ");
                taskloghex32(LOG_VERBOSE, cpuptr->opcode);
                tasklogreturn(LOG_VERBOSE);
                return taskseterror("Guru Error");

          case CPU_OPC :

        /* test interrupt signals et al. */
                if (cpuptr->stateflags) {
                  if (cpuptr->stateflags & CPU_STATERESET) {
                    cpuptr->stateflags = cpuptr->stateflags & ~(CPU_STATEHALT | CPU_STATERESET | CPU_STATEGURU | CPU_STATEBPT | CPU_STATEBRK);
                    cpuptr->step  = CPU_RESET_2;
                    return NULL;
                  }
                  if (cpuptr->stateflags & CPU_STATEHALT) {     // processor still halted?
                    screenupdate = 1;                           // delay loop
                    return "Processor halted";
                  }
                  if (cpuptr->stateflags & CPU_STATEBRK) {
                    cpuptr->stateflags = cpuptr->stateflags & ~CPU_STATEBRK;
                    cpuptr->step  = CPU_BRK_2;
                    return NULL;
                  }
                  if (cpuptr->stateflags & (CPU_STATENMI1 | CPU_STATENMI2)) {
                    cpuptr->stateflags = cpuptr->stateflags & ~(CPU_STATENMI1 | CPU_STATENMI2);
                    cpuptr->step  = CPU_NMI_2;
                    return NULL;
                  }
                  else {
                    if (cpuptr->stateflags & (CPU_STATEIRQ1 | CPU_STATEIRQ2)) {
                      if (!cpuptr->iflag) {
                        cpuptr->stateflags = cpuptr->stateflags & ~(CPU_STATEIRQ1 | CPU_STATEIRQ2);
                        cpuptr->step  = CPU_IRQ_2;
                        return NULL;
                      } // if (!cpuptr->iflag)
                    }
                  }
                  cpuptr->stateflags = cpuptr->stateflags &  ~(CPU_STATETRACE | CPU_STATEGURU | CPU_STATECOP);  // always remove these
                } // if (cpuptr->stateflags)

        /* test for breakpoint */
                if (cpuptr->pcreg == cpuptr->breakpoint) {
                  cpuptr->stateflags = cpuptr->stateflags ^ CPU_STATEBPT;
                  if (cpuptr->stateflags & CPU_STATEBPT) {
//                  setmessage("CPU stopped because breakpoint was reached");
//                  cpulight(0);
                    screenupdate = 1;
                    return "CPU stopped because breakpoint was reached";
                  }
                }

        /* fetch next opcode */
                (*cpuptr->rafptr).rafget(cpuptr->rafptr, cpuptr->pcreg, &(cpuptr->opcode));
                cpuptr->pcreg   = (cpuptr->pcreg + 1) & 0xffff;
                cpuptr->step    = cpu6502opcstep[cpuptr->opcode];
                break;


          case CPU_ZP_2 :
                (*cpuptr->rafptr).rafget(cpuptr->rafptr, cpuptr->pcreg, &data);
                cpuptr->addr    = data;
                cpuptr->pcreg   = (cpuptr->pcreg + 1) & 0xffff;
                cpuptr->step    = cpu6502opcstep2[cpuptr->opcode];
                break;


          case CPU_MZP_2 :
                (*cpuptr->rafptr).rafget(cpuptr->rafptr, cpuptr->pcreg, &data);                 /* zp */
                cpuptr->pcreg   = (cpuptr->pcreg + 1) & 0xffff;
                cpuptr->addr    = data;
                cpuptr->step    = CPU_MZP_3;
                break;
          case CPU_MZP_3 :
                (*cpuptr->rafptr).rafget(cpuptr->rafptr, cpuptr->addr, &cpuptr->data);          /* value */
                cpuptr->step    = CPU_MZP_4;
                break;
          case CPU_MZP_4 :
                (*cpuptr->rafptr).rafget(cpuptr->rafptr, (cpuptr->addr + 1) & 0xff, &data);     /* idle */
                cpuptr->step    = cpu6502opcstep2[cpuptr->opcode];
                break;


          case CPU_ZPX_2 :
                (*cpuptr->rafptr).rafget(cpuptr->rafptr, cpuptr->pcreg, &cpuptr->data);
                cpuptr->pcreg   = (cpuptr->pcreg + 1) & 0xffff;
                cpuptr->step    = CPU_ZPX_3;
                break;
          case CPU_ZPX_3 :
                cpuptr->addr    = (cpuptr->data + cpuptr->xreg) & 0xff;
                cpuptr->step    = cpu6502opcstep2[cpuptr->opcode];
                break;


          case CPU_MZPX_2 :
                (*cpuptr->rafptr).rafget(cpuptr->rafptr, cpuptr->pcreg, &cpuptr->data);         /* zp */
                cpuptr->step    = CPU_MZPX_3;
                break;
          case CPU_MZPX_3 :
                (*cpuptr->rafptr).rafget(cpuptr->rafptr, cpuptr->pcreg, &data);                 /* idle */
                cpuptr->addr    = (cpuptr->data + cpuptr->xreg) & 0xff;
                cpuptr->pcreg   = (cpuptr->pcreg + 1) & 0xffff;
                cpuptr->step    = CPU_MZPX_4;
                break;
          case CPU_MZPX_4 :
                (*cpuptr->rafptr).rafget(cpuptr->rafptr, cpuptr->addr, &cpuptr->data);          /* value */
                cpuptr->step    = CPU_MZPX_5;
                break;
          case CPU_MZPX_5 :
                (*cpuptr->rafptr).rafget(cpuptr->rafptr, (cpuptr->addr + 1) & 0xff, &data);     /* idle */
                cpuptr->step    = cpu6502opcstep2[cpuptr->opcode];
                break;


          case CPU_ZPY_2 :
                (*cpuptr->rafptr).rafget(cpuptr->rafptr, cpuptr->pcreg, &cpuptr->data);
                cpuptr->pcreg   = (cpuptr->pcreg + 1) & 0xffff;
                cpuptr->step    = CPU_ZPY_3;
                break;
          case CPU_ZPY_3 :
                cpuptr->addr    = (cpuptr->data + cpuptr->yreg) & 0xff;
                cpuptr->step    = cpu6502opcstep2[cpuptr->opcode];
                break;


          case CPU_ABS_2 :
                (*cpuptr->rafptr).rafget(cpuptr->rafptr, cpuptr->pcreg, &cpuptr->data);
                cpuptr->pcreg   = (cpuptr->pcreg + 1) & 0xffff;
                cpuptr->step    = CPU_ABS_3;
                break;
          case CPU_ABS_3 :
                (*cpuptr->rafptr).rafget(cpuptr->rafptr, cpuptr->pcreg, &data);
                cpuptr->pcreg   = (cpuptr->pcreg + 1) & 0xffff;
                cpuptr->addr    = cpuptr->data | (data << 8);
                cpuptr->step    = cpu6502opcstep2[cpuptr->opcode];
                break;


          case CPU_MABS_2 :
                (*cpuptr->rafptr).rafget(cpuptr->rafptr, cpuptr->pcreg, &cpuptr->data);         // addrl
                cpuptr->pcreg   = (cpuptr->pcreg + 1) & 0xffff;
                cpuptr->step    = CPU_MABS_3;
                break;
          case CPU_MABS_3 :
                (*cpuptr->rafptr).rafget(cpuptr->rafptr, cpuptr->pcreg, &data);                 // addrh
                cpuptr->pcreg   = (cpuptr->pcreg + 1) & 0xffff;
                cpuptr->addr    = cpuptr->data | (data << 8);
                cpuptr->step    = CPU_MABS_4;
                break;
          case CPU_MABS_4 :
                (*cpuptr->rafptr).rafget(cpuptr->rafptr, cpuptr->addr, &cpuptr->data);          // data
                cpuptr->step    = CPU_MABS_5;
                break;
          case CPU_MABS_5 :
                (*cpuptr->rafptr).rafget(cpuptr->rafptr, cpuptr->addr, &data);                  // idle
                cpuptr->step    = cpu6502opcstep2[cpuptr->opcode];
                break;


          case CPU_ABSX_2 :
                (*cpuptr->rafptr).rafget(cpuptr->rafptr, cpuptr->pcreg, &cpuptr->data);
                cpuptr->pcreg   = (cpuptr->pcreg + 1) & 0xffff;
                cpuptr->step    = CPU_ABSX_3;
                break;
          case CPU_ABSX_3 :
                (*cpuptr->rafptr).rafget(cpuptr->rafptr, cpuptr->pcreg, &data);
                cpuptr->pcreg   = (cpuptr->pcreg + 1) & 0xffff;
                cpuptr->addr    = cpuptr->data | (data << 8);
//              if (((cpuptr->addr & 0xff) + cpuptr->xreg) > 0xff) {
//                cpuptr->step  = CPU_ABSX_4;
//              }
//              else {
                  cpuptr->addr  = (cpuptr->addr + cpuptr->xreg) & 0xffff;
                  cpuptr->step  = cpu6502opcstep2[cpuptr->opcode];
//              }
                break;
          case CPU_ABSX_4 :
                (*cpuptr->rafptr).rafget(cpuptr->rafptr, cpuptr->addr, &data);                  /* idle */
                cpuptr->addr    = (cpuptr->addr + cpuptr->xreg) & 0xffff;
                cpuptr->step    = cpu6502opcstep2[cpuptr->opcode];
//              taskloghex32(cpuptr->pcreg - 3);
//              tasklogstring(": Extended ABS,x");
//              tasklogreturn();
                break;


          case CPU_MABSX_2 :
                (*cpuptr->rafptr).rafget(cpuptr->rafptr, cpuptr->pcreg, &cpuptr->data);         /* addrl */
                cpuptr->pcreg   = (cpuptr->pcreg + 1) & 0xffff;
                cpuptr->step    = CPU_MABSX_3;
                break;
          case CPU_MABSX_3 :
                (*cpuptr->rafptr).rafget(cpuptr->rafptr, cpuptr->pcreg, &data);                 /* addrh */
                cpuptr->pcreg   = (cpuptr->pcreg + 1) & 0xffff;
                cpuptr->addr    = cpuptr->data | (data << 8);
                cpuptr->step    = CPU_MABSX_4;
                break;
          case CPU_MABSX_4 :
                (*cpuptr->rafptr).rafget(cpuptr->rafptr,
                        (cpuptr->addr & 0xff00) | (((cpuptr->addr & 0xff) + cpuptr->xreg) & 0xff),
                        &data);                                                                 /* idle */
                cpuptr->addr    = (cpuptr->addr + cpuptr->xreg) & 0xffff;
                cpuptr->step    = CPU_MABSX_5;
                break;
          case CPU_MABSX_5 :
                (*cpuptr->rafptr).rafget(cpuptr->rafptr, cpuptr->addr, &cpuptr->data);          /* data */
                cpuptr->step    = CPU_MABSX_6;
                break;
          case CPU_MABSX_6 :
                (*cpuptr->rafptr).rafget(cpuptr->rafptr, (cpuptr->addr + 1) &0xffff, &data);    /* idle */
                cpuptr->step    = cpu6502opcstep2[cpuptr->opcode];
                break;


          case CPU_ABSY_2 :
                (*cpuptr->rafptr).rafget(cpuptr->rafptr, cpuptr->pcreg, &cpuptr->data);         /* absl */
                cpuptr->pcreg   = (cpuptr->pcreg + 1) & 0xffff;
                cpuptr->step    = CPU_ABSY_3;
                break;
          case CPU_ABSY_3 :
                (*cpuptr->rafptr).rafget(cpuptr->rafptr, cpuptr->pcreg, &data);                 /* absh */
                cpuptr->pcreg   = (cpuptr->pcreg + 1) & 0xffff;
                cpuptr->addr    = cpuptr->data | (data << 8);
//              if (((cpuptr->addr & 0xff) + cpuptr->yreg) > 0xff) {
//                cpuptr->addr  = ((cpuptr->addr + cpuptr->yreg) & 0xffff) - 0x100;
//                cpuptr->step  = CPU_ABSY_4;
//              }
//              else {
                  cpuptr->addr  = (cpuptr->addr + cpuptr->yreg) & 0xffff;
                  cpuptr->step  = cpu6502opcstep2[cpuptr->opcode];
//              }
                break;
          case CPU_ABSY_4 :
                (*cpuptr->rafptr).rafget(cpuptr->rafptr, cpuptr->addr, &data);                  // idle
                cpuptr->addr    = (cpuptr->addr + 0x100) & 0xffff;
                cpuptr->step    = cpu6502opcstep2[cpuptr->opcode];
                taskloghex32(LOG_VERBOSE, cpuptr->pcreg - 3);
                tasklogstring(LOG_VERBOSE, ": Extended ABS,y | address: ");
                taskloghex32(LOG_VERBOSE, cpuptr->addr);
                tasklogstring(LOG_VERBOSE, ", areg: ");
                taskloghex32(LOG_VERBOSE, cpuptr->areg);
                tasklogreturn(LOG_VERBOSE);
                break;


          case CPU_IZPY_2 :
                (*cpuptr->rafptr).rafget(cpuptr->rafptr, cpuptr->pcreg, &data);
                cpuptr->addr    = data;
                cpuptr->pcreg   = (cpuptr->pcreg + 1) & 0xffff;
                cpuptr->step    = CPU_IZPY_3;
                break;
          case CPU_IZPY_3 :
                (*cpuptr->rafptr).rafget(cpuptr->rafptr, cpuptr->addr, &cpuptr->data);
                cpuptr->step    = CPU_IZPY_4;
                break;
          case CPU_IZPY_4 :
                (*cpuptr->rafptr).rafget(cpuptr->rafptr, (cpuptr->addr + 1) & 0xff, &data);
                cpuptr->addr    = ((cpuptr->data | (data << 8)) + cpuptr->yreg) & 0xffff;
                cpuptr->step    = cpu6502opcstep2[cpuptr->opcode];
                break;


          case CPU_IZPX_2 :
                (*cpuptr->rafptr).rafget(cpuptr->rafptr, cpuptr->pcreg, &data);                 /* zp */
                cpuptr->addr    = (data + cpuptr->xreg) & 0xff;
                cpuptr->step    = CPU_IZPX_3;
                break;
          case CPU_IZPX_3 :
                (*cpuptr->rafptr).rafget(cpuptr->rafptr, cpuptr->pcreg, &data);                 /* idle */
                cpuptr->pcreg   = (cpuptr->pcreg + 1) & 0xffff;
                cpuptr->step    = CPU_IZPX_4;
                break;
          case CPU_IZPX_4 :
                (*cpuptr->rafptr).rafget(cpuptr->rafptr, cpuptr->addr, &cpuptr->data);          /* addrl */
                cpuptr->step    = CPU_IZPX_5;
                break;
          case CPU_IZPX_5 :
                (*cpuptr->rafptr).rafget(cpuptr->rafptr, (cpuptr->addr + 1) & 0xff, &data);     /* addrh */
                cpuptr->addr    = cpuptr->data | (data << 8);
                cpuptr->step    = cpu6502opcstep2[cpuptr->opcode];
                break;


          case CPU_RESET_2 :
                (*cpuptr->rafptr).rafget(cpuptr->rafptr, cpuptr->pcreg, &data);
                cpuptr->step    = CPU_RESET_3;
                break;
          case CPU_RESET_3 :
                (*cpuptr->rafptr).rafget(cpuptr->rafptr, cpuptr->spreg | 0x100, &data);
                cpuptr->spreg--;
                cpuptr->step    = CPU_RESET_4;
                break;
          case CPU_RESET_4 :
                (*cpuptr->rafptr).rafget(cpuptr->rafptr, cpuptr->spreg | 0x100, &data);
                cpuptr->spreg--;
                cpuptr->step    = CPU_RESET_5;
                break;
          case CPU_RESET_5 :
                (*cpuptr->rafptr).rafset(cpuptr->rafptr, cpuptr->spreg | 0x100, cpu6502getflags(cpuptr));
                cpuptr->spreg--;
                cpuptr->step    = CPU_RESET_6;
                break;
          case CPU_RESET_6 :
                (*cpuptr->rafptr).rafget(cpuptr->rafptr, 0xfffc, &cpuptr->data);
                cpuptr->step    = CPU_RESET_7;
                break;
          case CPU_RESET_7 :
                (*cpuptr->rafptr).rafget(cpuptr->rafptr, 0xfffd, &data);
                cpuptr->pcreg   = cpuptr->data | (data << 8);
                if (cpuptr->cputype & (CPU_65C02 | CPU_65SC02)) {
                  cpuptr->dflag = 0;                    // on a 65c02/65sc02 the dflag gets cleared on reset
                }
                cpuptr->step    = CPU_OPC;
                tasklogstring(LOG_BRIEF, "Reset 6502 to address ");
                taskloghex32(LOG_BRIEF, cpuptr->pcreg);
                tasklogreturn(LOG_BRIEF);
                break;


          case CPU_IRQ_2 :
//              tasklogstring("IRQ 6502 at address ");
//              taskloghex32(LOG_VERBOSE, cpuptr->pcreg);
//              tasklogreturn(LOG_VERBOSE);
                (*cpuptr->rafptr).rafget(cpuptr->rafptr, cpuptr->pcreg, &data);
                cpuptr->step    = CPU_IRQ_3;
                break;
          case CPU_IRQ_3 :
                (*cpuptr->rafptr).rafset(cpuptr->rafptr, cpuptr->spreg | 0x100, cpuptr->pcreg >> 8);
                cpuptr->spreg--;
                cpuptr->step    = CPU_IRQ_4;
                break;
          case CPU_IRQ_4 :
                (*cpuptr->rafptr).rafset(cpuptr->rafptr, cpuptr->spreg | 0x100, cpuptr->pcreg);
                cpuptr->spreg--;
                cpuptr->step    = CPU_IRQ_5;
                break;
          case CPU_IRQ_5 :
                (*cpuptr->rafptr).rafset(cpuptr->rafptr, cpuptr->spreg | 0x100, cpu6502getflags(cpuptr));
                cpuptr->spreg--;
                cpuptr->step    = CPU_IRQ_6;
                break;
          case CPU_IRQ_6 :
                (*cpuptr->rafptr).rafget(cpuptr->rafptr, 0xfffe, &cpuptr->data);
                cpuptr->step    = CPU_IRQ_7;
                break;
          case CPU_IRQ_7 :
                (*cpuptr->rafptr).rafget(cpuptr->rafptr, 0xffff, &data);
                cpuptr->pcreg   = cpuptr->data | (data << 8);
                cpuptr->iflag   = 1;
                if (cpuptr->cputype & (CPU_65C02 | CPU_65SC02)) {
                  cpuptr->dflag = 0;                    /* on a 65c02/65sc02 the dflag gets cleared on irq */
                }
                cpuptr->step    = CPU_OPC;
//              tasklogstring("IRQ 6502 to address ");
//              taskloghex32(cpuptr->pcreg);
//              tasklogreturn();
                break;


          case CPU_NMI_2 :
//              tasklogstring("NMI 6502 at address ");
//              taskloghex32(cpuptr->pcreg);
//              tasklogreturn();
                (*cpuptr->rafptr).rafget(cpuptr->rafptr, cpuptr->pcreg, &data);
                cpuptr->step    = CPU_NMI_3;
                break;
          case CPU_NMI_3 :
                (*cpuptr->rafptr).rafset(cpuptr->rafptr, cpuptr->spreg | 0x100, cpuptr->pcreg >> 8);
                cpuptr->spreg--;
                cpuptr->step    = CPU_NMI_4;
                break;
          case CPU_NMI_4 :
                (*cpuptr->rafptr).rafset(cpuptr->rafptr, cpuptr->spreg | 0x100, cpuptr->pcreg);
                cpuptr->spreg--;
                cpuptr->step    = CPU_NMI_5;
                break;
          case CPU_NMI_5 :
                (*cpuptr->rafptr).rafset(cpuptr->rafptr, cpuptr->spreg | 0x100, cpu6502getflags(cpuptr));
                cpuptr->spreg--;
                cpuptr->step    = CPU_NMI_6;
                break;
          case CPU_NMI_6 :
                (*cpuptr->rafptr).rafget(cpuptr->rafptr, 0xfffa, &cpuptr->data);
                cpuptr->step    = CPU_NMI_7;
                break;
          case CPU_NMI_7 :
                (*cpuptr->rafptr).rafget(cpuptr->rafptr, 0xfffb, &data);
                cpuptr->pcreg   = cpuptr->data | (data << 8);
                cpuptr->iflag   = 1;
                if (cpuptr->cputype & (CPU_65C02 | CPU_65SC02)) {
                  cpuptr->dflag = 0;                    /* on a 65c02/65sc02 the dflag gets cleared on nmi */
                }
                cpuptr->step    = CPU_OPC;
//              tasklogstring("NMI 6502 to address ");
//              taskloghex32(cpuptr->pcreg);
//              tasklogreturn();
                break;


          case CPU_LDAI_2 :
                (*cpuptr->rafptr).rafget(cpuptr->rafptr, cpuptr->pcreg, &cpuptr->areg);
                cpuptr->pcreg   = (cpuptr->pcreg + 1) & 0xffff;
                cpu6502setflagsnz(cpuptr, cpuptr->areg);
                cpuptr->step    = CPU_OPC;
                break;
          case CPU_LDA :
                (*cpuptr->rafptr).rafget(cpuptr->rafptr, cpuptr->addr, &cpuptr->areg);
                cpu6502setflagsnz(cpuptr, cpuptr->areg);
                cpuptr->step    = CPU_OPC;
//              tasklogstring("LDA | address: ");
//              taskloghex32(cpuptr->addr);
//              tasklogstring(", areg: ");
//              taskloghex32(cpuptr->areg);
//              tasklogreturn();
                break;
          case CPU_STA :
                (*cpuptr->rafptr).rafset(cpuptr->rafptr, cpuptr->addr, cpuptr->areg);
                cpuptr->step    = CPU_OPC;
//              tasklogstring("STA | address: ");
//              taskloghex32(cpuptr->addr);
//              tasklogstring(", areg: ");
//              taskloghex32(cpuptr->areg);
//              tasklogreturn();
                break;
          case CPU_CMPI_2 :
                (*cpuptr->rafptr).rafget(cpuptr->rafptr, cpuptr->pcreg, &data);
                cpuptr->pcreg   = (cpuptr->pcreg + 1) & 0xffff;
                cpuptr->cflag =   (cpuptr->areg >= data)                ? 1 : 0;
                cpuptr->nflag = (((cpuptr->areg  - data) & 0x80) == 0)  ? 0 : 1;
                cpuptr->zflag =   (cpuptr->areg == data)                ? 1 : 0;
                cpuptr->step    = CPU_OPC;
                break;
          case CPU_CMP :
                (*cpuptr->rafptr).rafget(cpuptr->rafptr, cpuptr->addr, &data);
                cpuptr->cflag =   (cpuptr->areg >= data)                ? 1 : 0;
                cpuptr->nflag = (((cpuptr->areg  - data) & 0x80) == 0)  ? 0 : 1;
                cpuptr->zflag =   (cpuptr->areg == data)                ? 1 : 0;
                cpuptr->step    = CPU_OPC;
//              tasklogstring("CMP | address: ");
//              taskloghex32(cpuptr->addr);
//              tasklogstring(", areg: ");
//              taskloghex32(cpuptr->areg);
//              tasklogstring(", value: ");
//              taskloghex32(data);
//              cpu6502flagslog(cpuptr);
//              tasklogreturn();
                break;


          case CPU_LDXI_2 :
                (*cpuptr->rafptr).rafget(cpuptr->rafptr, cpuptr->pcreg, &cpuptr->xreg);
                cpuptr->pcreg   = (cpuptr->pcreg + 1) & 0xffff;
                cpu6502setflagsnz(cpuptr, cpuptr->xreg);
                cpuptr->step    = CPU_OPC;
//              taskloghex32(cpuptr->pcreg - 2);
//              tasklogstring(": LDX #");
//              tasklogreturn();
                break;
          case CPU_LDX :
                (*cpuptr->rafptr).rafget(cpuptr->rafptr, cpuptr->addr, &cpuptr->xreg);
                cpu6502setflagsnz(cpuptr, cpuptr->xreg);
                cpuptr->step    = CPU_OPC;
                break;
          case CPU_STX :
                (*cpuptr->rafptr).rafset(cpuptr->rafptr, cpuptr->addr, cpuptr->xreg);
                cpuptr->step    = CPU_OPC;
                break;
          case CPU_CPXI_2 :
                (*cpuptr->rafptr).rafget(cpuptr->rafptr, cpuptr->pcreg, &data);
                cpuptr->pcreg   = (cpuptr->pcreg + 1) & 0xffff;
                cpuptr->cflag =   (cpuptr->xreg >= data)                ? 1 : 0;
                cpuptr->nflag = (((cpuptr->xreg  - data) & 0x80) == 0)  ? 0 : 1;
                cpuptr->zflag =   (cpuptr->xreg == data)                ? 1 : 0;
                cpuptr->step    = CPU_OPC;
                break;
          case CPU_CPX :
                (*cpuptr->rafptr).rafget(cpuptr->rafptr, cpuptr->addr, &data);
                cpuptr->cflag =   (cpuptr->xreg >= data)                ? 1 : 0;
                cpuptr->nflag = (((cpuptr->xreg  - data) & 0x80) == 0)  ? 0 : 1;
                cpuptr->zflag =   (cpuptr->xreg == data)                ? 1 : 0;
                cpuptr->step    = CPU_OPC;
//              taskloghex32(cpuptr->pcreg);
//              tasklogstring(": CPX done");
//              tasklogreturn();
                break;


          case CPU_LDYI_2 :
                (*cpuptr->rafptr).rafget(cpuptr->rafptr, cpuptr->pcreg, &cpuptr->yreg);
                cpuptr->pcreg   = (cpuptr->pcreg + 1) & 0xffff;
                cpu6502setflagsnz(cpuptr, cpuptr->yreg);
                cpuptr->step    = CPU_OPC;
//              taskloghex32(cpuptr->pcreg - 2);
//              tasklogstring(": LDY #");
//              tasklogreturn();
                break;
          case CPU_LDY :
                (*cpuptr->rafptr).rafget(cpuptr->rafptr, cpuptr->addr, &cpuptr->yreg);
                cpu6502setflagsnz(cpuptr, cpuptr->yreg);
                cpuptr->step    = CPU_OPC;
                break;
          case CPU_STY :
                (*cpuptr->rafptr).rafset(cpuptr->rafptr, cpuptr->addr, cpuptr->yreg);
                cpuptr->step    = CPU_OPC;
                break;
          case CPU_CPYI_2 :
                (*cpuptr->rafptr).rafget(cpuptr->rafptr, cpuptr->pcreg, &data);
                cpuptr->pcreg   = (cpuptr->pcreg + 1) & 0xffff;
                cpuptr->cflag =   (cpuptr->yreg >= data)                ? 1 : 0;
                cpuptr->nflag = (((cpuptr->yreg  - data) & 0x80) == 0)  ? 0 : 1;
                cpuptr->zflag =   (cpuptr->yreg == data)                ? 1 : 0;
                cpuptr->step    = CPU_OPC;
                break;
          case CPU_CPY :
                (*cpuptr->rafptr).rafget(cpuptr->rafptr, cpuptr->addr, &data);
                cpuptr->cflag =   (cpuptr->yreg >= data)                ? 1 : 0;
                cpuptr->nflag = (((cpuptr->yreg  - data) & 0x80) == 0)  ? 0 : 1;
                cpuptr->zflag =   (cpuptr->yreg == data)                ? 1 : 0;
                cpuptr->step    = CPU_OPC;
//              taskloghex32(cpuptr->pcreg);
//              tasklogstring(": CPY done");
//              tasklogreturn();
                break;


          case CPU_BIT :
                (*cpuptr->rafptr).rafget(cpuptr->rafptr, cpuptr->addr, &data);
                cpuptr->nflag   = (data & 0x80) ? 1 : 0;
                cpuptr->vflag   = (data & 0x40) ? 1 : 0;
                cpuptr->zflag   = (data & cpuptr->areg) ? 0 : 1;
                cpuptr->step    = CPU_OPC;
                break;


          case CPU_ANDI_2 :
                (*cpuptr->rafptr).rafget(cpuptr->rafptr, cpuptr->pcreg, &data);
                cpuptr->pcreg   = (cpuptr->pcreg + 1) & 0xffff;
                cpuptr->areg    = cpuptr->areg & data;
                cpu6502setflagsnz(cpuptr, cpuptr->areg);
                cpuptr->step    = CPU_OPC;
                break;
          case CPU_AND :
                (*cpuptr->rafptr).rafget(cpuptr->rafptr, cpuptr->addr, &data);
                cpuptr->areg    = cpuptr->areg & data;
                cpu6502setflagsnz(cpuptr, cpuptr->areg);
                cpuptr->step    = CPU_OPC;
                break;
          case CPU_EORI_2 :
                (*cpuptr->rafptr).rafget(cpuptr->rafptr, cpuptr->pcreg, &data);
                cpuptr->pcreg   = (cpuptr->pcreg + 1) & 0xffff;
                cpuptr->areg    = cpuptr->areg ^ data;
                cpu6502setflagsnz(cpuptr, cpuptr->areg);
                cpuptr->step    = CPU_OPC;
                break;
          case CPU_EOR :
                (*cpuptr->rafptr).rafget(cpuptr->rafptr, cpuptr->addr, &data);
                cpuptr->areg    = cpuptr->areg ^ data;
                cpu6502setflagsnz(cpuptr, cpuptr->areg);
                cpuptr->step    = CPU_OPC;
//              tasklogstring("EOR | value: ");
//              taskloghex32(data);
//              tasklogreturn();
                break;
          case CPU_ORAI_2 :
                (*cpuptr->rafptr).rafget(cpuptr->rafptr, cpuptr->pcreg, &data);
                cpuptr->pcreg   = (cpuptr->pcreg + 1) & 0xffff;
                cpuptr->areg    = cpuptr->areg | data;
                cpu6502setflagsnz(cpuptr, cpuptr->areg);
                cpuptr->step    = CPU_OPC;
                break;
          case CPU_ORA :
                (*cpuptr->rafptr).rafget(cpuptr->rafptr, cpuptr->addr, &data);
                cpuptr->areg    = cpuptr->areg | data;
                cpu6502setflagsnz(cpuptr, cpuptr->areg);
                cpuptr->step    = CPU_OPC;
                break;


          case CPU_ADCI_2 :
                (*cpuptr->rafptr).rafget(cpuptr->rafptr, cpuptr->pcreg, &data);
                cpuptr->pcreg   = (cpuptr->pcreg + 1) & 0xffff;
                cpu6502adc(cpuptr, data);
                cpuptr->step    = CPU_OPC;
                break;
          case CPU_ADC :
                (*cpuptr->rafptr).rafget(cpuptr->rafptr, cpuptr->addr, &data);
                cpu6502adc(cpuptr, data);
                cpuptr->step    = CPU_OPC;
                break;
          case CPU_SBCI_2 :
                (*cpuptr->rafptr).rafget(cpuptr->rafptr, cpuptr->pcreg, &data);
                cpuptr->pcreg   = (cpuptr->pcreg + 1) & 0xffff;
                cpu6502sbc(cpuptr, data);
                cpuptr->step    = CPU_OPC;
                break;
          case CPU_SBC :
                (*cpuptr->rafptr).rafget(cpuptr->rafptr, cpuptr->addr, &data);
                cpu6502sbc(cpuptr, data);
                cpuptr->step    = CPU_OPC;
                break;


          case CPU_DEA_2 :
                (*cpuptr->rafptr).rafget(cpuptr->rafptr, cpuptr->pcreg, &data);         /* idle */
                cpuptr->areg--;
                cpu6502setflagsnz(cpuptr, cpuptr->areg);
                cpuptr->step    = CPU_OPC;
                break;
          case CPU_DEX_2 :
                (*cpuptr->rafptr).rafget(cpuptr->rafptr, cpuptr->pcreg, &data);         /* idle */
                cpuptr->xreg--;
                cpu6502setflagsnz(cpuptr, cpuptr->xreg);
                cpuptr->step    = CPU_OPC;
                break;
          case CPU_DEY_2 :
                (*cpuptr->rafptr).rafget(cpuptr->rafptr, cpuptr->pcreg, &data);         /* idle */
                cpuptr->yreg--;
                cpu6502setflagsnz(cpuptr, cpuptr->yreg);
                cpuptr->step    = CPU_OPC;
                break;
          case CPU_DEC :
                cpuptr->data--;
                cpu6502setflagsnz(cpuptr, cpuptr->data);
                (*cpuptr->rafptr).rafset(cpuptr->rafptr, cpuptr->addr, cpuptr->data);
                cpuptr->step    = CPU_OPC;
                break;


          case CPU_INA_2 :
                (*cpuptr->rafptr).rafget(cpuptr->rafptr, cpuptr->pcreg, &data);         /* idle */
                cpuptr->areg++;
                cpu6502setflagsnz(cpuptr, cpuptr->areg);
                cpuptr->step    = CPU_OPC;
                break;
          case CPU_INX_2 :
                (*cpuptr->rafptr).rafget(cpuptr->rafptr, cpuptr->pcreg, &data);         /* idle */
                cpuptr->xreg++;
                cpu6502setflagsnz(cpuptr, cpuptr->xreg);
                cpuptr->step    = CPU_OPC;
                break;
          case CPU_INY_2 :
                (*cpuptr->rafptr).rafget(cpuptr->rafptr, cpuptr->pcreg, &data);         /* idle */
                cpuptr->yreg++;
                cpu6502setflagsnz(cpuptr, cpuptr->yreg);
                cpuptr->step    = CPU_OPC;
//              taskloghex32(cpuptr->pcreg - 1);
//              tasklogstring(": INY | yreg: ");
//              taskloghex32(cpuptr->yreg);
//              cpu6502flagslog(cpuptr);
//              tasklogreturn();
                break;
          case CPU_INC :
                cpuptr->data++;
                cpu6502setflagsnz(cpuptr, cpuptr->data);
                (*cpuptr->rafptr).rafset(cpuptr->rafptr, cpuptr->addr, cpuptr->data);
                cpuptr->step    = CPU_OPC;
                break;


          case CPU_TAX_2 :
                (*cpuptr->rafptr).rafget(cpuptr->rafptr, cpuptr->pcreg, &data);         /* idle */
                cpuptr->xreg    = cpuptr->areg;
                cpu6502setflagsnz(cpuptr, cpuptr->xreg);
                cpuptr->step    = CPU_OPC;
                break;
          case CPU_TAY_2 :
                (*cpuptr->rafptr).rafget(cpuptr->rafptr, cpuptr->pcreg, &data);         /* idle */
                cpuptr->yreg    = cpuptr->areg;
                cpu6502setflagsnz(cpuptr, cpuptr->yreg);
                cpuptr->step    = CPU_OPC;
                break;
          case CPU_TXA_2 :
                (*cpuptr->rafptr).rafget(cpuptr->rafptr, cpuptr->pcreg, &data);         /* idle */
                cpuptr->areg    = cpuptr->xreg;
                cpu6502setflagsnz(cpuptr, cpuptr->areg);
                cpuptr->step    = CPU_OPC;
                break;
          case CPU_TYA_2 :
                (*cpuptr->rafptr).rafget(cpuptr->rafptr, cpuptr->pcreg, &data);         /* idle */
                cpuptr->areg    = cpuptr->yreg;
                cpu6502setflagsnz(cpuptr, cpuptr->areg);
                cpuptr->step    = CPU_OPC;
                break;
          case CPU_TXS_2 :
                (*cpuptr->rafptr).rafget(cpuptr->rafptr, cpuptr->pcreg, &data);         /* idle */
                cpuptr->spreg   = cpuptr->xreg;
                cpuptr->step    = CPU_OPC;
                break;
          case CPU_TSX_2 :
                (*cpuptr->rafptr).rafget(cpuptr->rafptr, cpuptr->pcreg, &data);         /* idle */
                cpuptr->xreg    = cpuptr->spreg;
                cpu6502setflagsnz(cpuptr, cpuptr->xreg);
                cpuptr->step    = CPU_OPC;
                break;


          case CPU_PHA_2 :
                (*cpuptr->rafptr).rafget(cpuptr->rafptr, cpuptr->pcreg, &data);         /* idle */
                cpuptr->step    = CPU_PHA_3;
                break;
          case CPU_PHA_3 :
                (*cpuptr->rafptr).rafset(cpuptr->rafptr, cpuptr->spreg | 0x100, cpuptr->areg);
                cpuptr->spreg--;
                cpuptr->step    = CPU_OPC;
                break;
          case CPU_PHX_2 :
                (*cpuptr->rafptr).rafget(cpuptr->rafptr, cpuptr->pcreg, &data);         /* idle */
                cpuptr->step    = CPU_PHX_3;
                break;
          case CPU_PHX_3 :
                (*cpuptr->rafptr).rafset(cpuptr->rafptr, cpuptr->spreg | 0x100, cpuptr->xreg);
                cpuptr->spreg--;
                cpuptr->step    = CPU_OPC;
                break;
          case CPU_PHY_2 :
                (*cpuptr->rafptr).rafget(cpuptr->rafptr, cpuptr->pcreg, &data);         /* idle */
                cpuptr->step    = CPU_PHY_3;
                break;
          case CPU_PHY_3 :
                (*cpuptr->rafptr).rafset(cpuptr->rafptr, cpuptr->spreg | 0x100, cpuptr->yreg);
                cpuptr->spreg--;
                cpuptr->step    = CPU_OPC;
                break;
          case CPU_PHP_2 :
                (*cpuptr->rafptr).rafget(cpuptr->rafptr, cpuptr->pcreg, &data);
                cpuptr->step    = CPU_PHP_3;
                break;
          case CPU_PHP_3 :
                (*cpuptr->rafptr).rafset(cpuptr->rafptr, cpuptr->spreg | 0x100, cpu6502getflags(cpuptr) | 0x10);
                cpuptr->spreg--;
                cpuptr->step    = CPU_OPC;
                break;


          case CPU_PLA_2 :
                (*cpuptr->rafptr).rafget(cpuptr->rafptr, cpuptr->pcreg, &data);                 /* idle */
                cpuptr->step    = CPU_PLA_3;
                break;
          case CPU_PLA_3 :
                (*cpuptr->rafptr).rafget(cpuptr->rafptr, cpuptr->spreg | 0x100, &data);         /* idle */
                cpuptr->spreg++;
                cpuptr->step    = CPU_PLA_4;
                break;
          case CPU_PLA_4 :
                (*cpuptr->rafptr).rafget(cpuptr->rafptr, cpuptr->spreg | 0x100, &cpuptr->areg);
                cpu6502setflagsnz(cpuptr, cpuptr->areg);
                cpuptr->step    = CPU_OPC;
                break;
          case CPU_PLX_2 :
                (*cpuptr->rafptr).rafget(cpuptr->rafptr, cpuptr->pcreg, &data);                 /* idle */
                cpuptr->step    = CPU_PLX_3;
                break;
          case CPU_PLX_3 :
                (*cpuptr->rafptr).rafget(cpuptr->rafptr, cpuptr->spreg | 0x100, &data);         /* idle */
                cpuptr->spreg++;
                cpuptr->step    = CPU_PLX_4;
                break;
          case CPU_PLX_4 :
                (*cpuptr->rafptr).rafget(cpuptr->rafptr, cpuptr->spreg | 0x100, &cpuptr->xreg);
                cpu6502setflagsnz(cpuptr, cpuptr->xreg);
                cpuptr->step    = CPU_OPC;
                break;
          case CPU_PLY_2 :
                (*cpuptr->rafptr).rafget(cpuptr->rafptr, cpuptr->pcreg, &data);                 /* idle */
                cpuptr->step    = CPU_PLY_3;
                break;
          case CPU_PLY_3 :
                (*cpuptr->rafptr).rafget(cpuptr->rafptr, cpuptr->spreg | 0x100, &data);         /* idle */
                cpuptr->spreg++;
                cpuptr->step    = CPU_PLY_4;
                break;
          case CPU_PLY_4 :
                (*cpuptr->rafptr).rafget(cpuptr->rafptr, cpuptr->spreg | 0x100, &cpuptr->yreg);
                cpu6502setflagsnz(cpuptr, cpuptr->yreg);
                cpuptr->step    = CPU_OPC;
                break;
          case CPU_PLP_2 :
                (*cpuptr->rafptr).rafget(cpuptr->rafptr, cpuptr->pcreg, &data);                 /* idle */
                cpuptr->step    = CPU_PLP_3;
                break;
          case CPU_PLP_3 :
                (*cpuptr->rafptr).rafget(cpuptr->rafptr, cpuptr->spreg | 0x100, &data);         /* idle */
                cpuptr->spreg++;
                cpuptr->step    = CPU_PLP_4;
                break;
          case CPU_PLP_4 :
                (*cpuptr->rafptr).rafget(cpuptr->rafptr, cpuptr->spreg | 0x100, &data);
                cpu6502setflags(cpuptr, data);
                cpuptr->step    = CPU_OPC;
                break;


          case CPU_CLC_2 :
                (*cpuptr->rafptr).rafget(cpuptr->rafptr, cpuptr->pcreg, &data);                 /* idle */
                cpuptr->cflag   = 0;
                cpuptr->step    = CPU_OPC;
                break;
          case CPU_SEC_2 :
                (*cpuptr->rafptr).rafget(cpuptr->rafptr, cpuptr->pcreg, &data);                 /* idle */
                cpuptr->cflag   = 1;
                cpuptr->step    = CPU_OPC;
                break;
          case CPU_CLI_2 :
                (*cpuptr->rafptr).rafget(cpuptr->rafptr, cpuptr->pcreg, &data);                 /* idle */
                cpuptr->iflag   = 0;
                cpuptr->step    = CPU_OPC;
                break;
          case CPU_SEI_2 :
                (*cpuptr->rafptr).rafget(cpuptr->rafptr, cpuptr->pcreg, &data);                 /* idle */
                cpuptr->iflag   = 1;
                cpuptr->step    = CPU_OPC;
                break;
          case CPU_CLD_2 :
                (*cpuptr->rafptr).rafget(cpuptr->rafptr, cpuptr->pcreg, &data);                 /* idle */
                cpuptr->dflag   = 0;
                cpuptr->step    = CPU_OPC;
//              taskloghex32(cpuptr->pcreg - 1);
//              tasklogstring(": CLD");
//              tasklogreturn();
                break;
          case CPU_SED_2 :
                (*cpuptr->rafptr).rafget(cpuptr->rafptr, cpuptr->pcreg, &data);                 /* idle */
                cpuptr->dflag   = 1;
                cpuptr->step    = CPU_OPC;
                break;
          case CPU_CLV_2 :
                (*cpuptr->rafptr).rafget(cpuptr->rafptr, cpuptr->pcreg, &data);                 /* idle */
                cpuptr->vflag   = 0;
                cpuptr->step    = CPU_OPC;
                break;


          case CPU_NOP_2 :
                (*cpuptr->rafptr).rafget(cpuptr->rafptr, cpuptr->pcreg, &data);                 /* idle */
                cpuptr->step    = CPU_OPC;
                break;


          case CPU_ASL_2 :
                cpuptr->cflag   = (cpuptr->areg & 0x80) ? 1 : 0;
                cpuptr->areg    = cpuptr->areg << 1;
                cpu6502setflagsnz(cpuptr, cpuptr->areg);
                cpuptr->step    = CPU_OPC;
                break;
          case CPU_ASL :
                cpuptr->cflag   = (cpuptr->data & 0x80) ? 1 : 0;
                cpuptr->data    = cpuptr->data << 1;
                cpu6502setflagsnz(cpuptr, cpuptr->data);
                (*cpuptr->rafptr).rafset(cpuptr->rafptr, cpuptr->addr, cpuptr->data);
                cpuptr->step    = CPU_OPC;
                break;
          case CPU_LSR_2 :
                cpuptr->cflag   = (cpuptr->areg & 0x01) ? 1 : 0;
                cpuptr->areg    = cpuptr->areg >> 1;
                cpu6502setflagsnz(cpuptr, cpuptr->areg);
                cpuptr->step    = CPU_OPC;
                break;
          case CPU_LSR :
                cpuptr->cflag   = (cpuptr->data & 0x01) ? 1 : 0;
                cpuptr->data    = cpuptr->data >> 1;
                cpu6502setflagsnz(cpuptr, cpuptr->data);
                (*cpuptr->rafptr).rafset(cpuptr->rafptr, cpuptr->addr, cpuptr->data);
                cpuptr->step    = CPU_OPC;
                break;
          case CPU_ROL_2 :
                if (cpuptr->cflag) {
                  cpuptr->cflag = (cpuptr->areg & 0x80) ? 1 : 0;
                  cpuptr->areg  = (cpuptr->areg << 1) | 1;
                }
                else {
                  cpuptr->cflag = (cpuptr->areg & 0x80) ? 1 : 0;
                  cpuptr->areg  = cpuptr->areg << 1;
                }
                cpu6502setflagsnz(cpuptr, cpuptr->areg);
                cpuptr->step    = CPU_OPC;
                break;
          case CPU_ROL :
                if (cpuptr->cflag) {
                  cpuptr->cflag = (cpuptr->data & 0x80) ? 1 : 0;
                  cpuptr->data  = (cpuptr->data << 1) | 1;
                }
                else {
                  cpuptr->cflag = (cpuptr->data & 0x80) ? 1 : 0;
                  cpuptr->data  = cpuptr->data << 1;
                }
                cpu6502setflagsnz(cpuptr, cpuptr->data);
                (*cpuptr->rafptr).rafset(cpuptr->rafptr, cpuptr->addr, cpuptr->data);
                cpuptr->step    = CPU_OPC;
                break;
          case CPU_ROR_2 :
                if (cpuptr->cflag) {
                  cpuptr->cflag = (cpuptr->areg & 0x01) ? 1 : 0;
                  cpuptr->areg  = (cpuptr->areg >> 1) | 0x80;
                }
                else {
                  cpuptr->cflag = (cpuptr->areg & 0x01) ? 1 : 0;
                  cpuptr->areg  = cpuptr->areg >> 1;
                }
                cpu6502setflagsnz(cpuptr, cpuptr->areg);
                cpuptr->step    = CPU_OPC;
                break;
          case CPU_ROR :
                if (cpuptr->cflag) {
                  cpuptr->cflag = (cpuptr->data & 0x01) ? 1 : 0;
                  cpuptr->data  = (cpuptr->data >> 1) | 0x80;
                }
                else {
                  cpuptr->cflag = (cpuptr->data & 0x01) ? 1 : 0;
                  cpuptr->data  = cpuptr->data >> 1;
                }
                cpu6502setflagsnz(cpuptr, cpuptr->data);
                (*cpuptr->rafptr).rafset(cpuptr->rafptr, cpuptr->addr, cpuptr->data);
                cpuptr->step    = CPU_OPC;
                break;


          case CPU_BPL_2 :
                (*cpuptr->rafptr).rafget(cpuptr->rafptr, cpuptr->pcreg, &cpuptr->data);
                cpuptr->pcreg   = (cpuptr->pcreg + 1) & 0xffff;
                if (cpuptr->nflag) {
                  cpuptr->step  = CPU_OPC;
                }
                else {
                  cpuptr->step  = CPU_BRA_3;
                }
                break;
          case CPU_BMI_2 :
                (*cpuptr->rafptr).rafget(cpuptr->rafptr, cpuptr->pcreg, &cpuptr->data);
                cpuptr->pcreg   = (cpuptr->pcreg + 1) & 0xffff;
                if (cpuptr->nflag) {
                  cpuptr->step  = CPU_BRA_3;
                }
                else {
                  cpuptr->step  = CPU_OPC;
                }
                break;
          case CPU_BNE_2 :
                (*cpuptr->rafptr).rafget(cpuptr->rafptr, cpuptr->pcreg, &cpuptr->data);
                cpuptr->pcreg   = (cpuptr->pcreg + 1) & 0xffff;
                if (cpuptr->zflag) {
                  cpuptr->step  = CPU_OPC;
                }
                else {
                  cpuptr->step  = CPU_BRA_3;
                }
                break;
          case CPU_BEQ_2 :
                (*cpuptr->rafptr).rafget(cpuptr->rafptr, cpuptr->pcreg, &cpuptr->data);
                cpuptr->pcreg   = (cpuptr->pcreg + 1) & 0xffff;
                if (cpuptr->zflag) {
                  cpuptr->step  = CPU_BRA_3;
                }
                else {
                  cpuptr->step  = CPU_OPC;
                }
                break;
          case CPU_BCC_2 :
                (*cpuptr->rafptr).rafget(cpuptr->rafptr, cpuptr->pcreg, &cpuptr->data);
                cpuptr->pcreg   = (cpuptr->pcreg + 1) & 0xffff;
                if (cpuptr->cflag) {
                  cpuptr->step  = CPU_OPC;
                }
                else {
                  cpuptr->step  = CPU_BRA_3;
                }
                break;
          case CPU_BCS_2 :
                (*cpuptr->rafptr).rafget(cpuptr->rafptr, cpuptr->pcreg, &cpuptr->data);
                cpuptr->pcreg   = (cpuptr->pcreg + 1) & 0xffff;
                if (cpuptr->cflag) {
                  cpuptr->step  = CPU_BRA_3;
                }
                else {
                  cpuptr->step  = CPU_OPC;
                }
                break;
          case CPU_BVC_2 :
                (*cpuptr->rafptr).rafget(cpuptr->rafptr, cpuptr->pcreg, &cpuptr->data);
                cpuptr->pcreg   = (cpuptr->pcreg + 1) & 0xffff;
                if (cpuptr->vflag) {
                  cpuptr->step  = CPU_OPC;
                }
                else {
                  cpuptr->step  = CPU_BRA_3;
                }
                break;
          case CPU_BVS_2 :
                (*cpuptr->rafptr).rafget(cpuptr->rafptr, cpuptr->pcreg, &cpuptr->data);
                cpuptr->pcreg   = (cpuptr->pcreg + 1) & 0xffff;
                if (cpuptr->vflag) {
                  cpuptr->step  = CPU_BRA_3;
                }
                else {
                  cpuptr->step  = CPU_OPC;
                }
                break;
          case CPU_BRA_2 :
                (*cpuptr->rafptr).rafget(cpuptr->rafptr, cpuptr->pcreg, &cpuptr->data);
                cpuptr->pcreg   = (cpuptr->pcreg + 1) & 0xffff;
                cpuptr->step    = CPU_BRA_3;
                break;
          case CPU_BRA_3 :
//              taskloghex32(cpuptr->pcreg - 2);
                (*cpuptr->rafptr).rafget(cpuptr->rafptr, cpuptr->pcreg, &data);
                cpuptr->addr  = (cpuptr->pcreg & 0xff) + cpuptr->data;
                cpuptr->pcreg = (cpuptr->pcreg & 0xff00) | (cpuptr->addr & 0xff);
                if (cpuptr->data & 0x80) {
                  if (cpuptr->addr < 0x100) {
                    cpuptr->step        = CPU_BRAM_4;
                  }
                  else {
                    cpuptr->step        = CPU_OPC;
//                  tasklogstring(": Branch taken to address ");
//                  taskloghex32(cpuptr->pcreg);
//                  tasklogreturn();
                  }
                }
                else {
                  if (cpuptr->addr >= 0x100) {
                    cpuptr->step        = CPU_BRAP_4;
                  }
                  else {
                    cpuptr->step        = CPU_OPC;
//                  tasklogstring(": Branch taken to address ");
//                  taskloghex32(cpuptr->pcreg);
//                  tasklogreturn();
                  }
                }
                break;
          case CPU_BRAM_4 :
                (*cpuptr->rafptr).rafget(cpuptr->rafptr, cpuptr->pcreg, &data);
                cpuptr->pcreg   = (cpuptr->pcreg - 0x100) & 0xffff;
                cpuptr->step    = CPU_OPC;
//              tasklogstring(": Extended branch taken to address ");
//              taskloghex32(cpuptr->pcreg);
//              tasklogreturn();
                break;
          case CPU_BRAP_4 :
                (*cpuptr->rafptr).rafget(cpuptr->rafptr, cpuptr->pcreg, &data);
                cpuptr->pcreg   = (cpuptr->pcreg + 0x100) & 0xffff;
                cpuptr->step    = CPU_OPC;
//              tasklogstring(": Extended branch taken to address ");
//              taskloghex32(cpuptr->pcreg);
//              tasklogreturn();
                break;


          case CPU_JMPABS_2 :
//              taskloghex32(cpuptr->pcreg);
//              tasklogstring(": JMP to address ");
                (*cpuptr->rafptr).rafget(cpuptr->rafptr, cpuptr->pcreg, &cpuptr->data);
                cpuptr->pcreg   = (cpuptr->pcreg + 1) & 0xffff;
                cpuptr->step    = CPU_JMPABS_3;
                break;
          case CPU_JMPABS_3 :
                (*cpuptr->rafptr).rafget(cpuptr->rafptr, cpuptr->pcreg, &data);
                cpuptr->pcreg   = cpuptr->data | (data << 8);
                cpuptr->step    = CPU_OPC;
//              taskloghex32(cpuptr->pcreg);
//              tasklogreturn();
                break;


          case CPU_JMPIABS_2 :
//              taskloghex32(cpuptr->pcreg);
//              tasklogstring(": JMP indirect to address ");
                (*cpuptr->rafptr).rafget(cpuptr->rafptr, cpuptr->pcreg, &cpuptr->data);
                cpuptr->pcreg   = (cpuptr->pcreg + 1) & 0xffff;
                cpuptr->step    = CPU_JMPIABS_3;
                break;
          case CPU_JMPIABS_3 :
                (*cpuptr->rafptr).rafget(cpuptr->rafptr, cpuptr->pcreg, &data);
                cpuptr->addr    = cpuptr->data | (data << 8);
                cpuptr->step    = CPU_JMPIABS_4;
                break;
          case CPU_JMPIABS_4 :
                (*cpuptr->rafptr).rafget(cpuptr->rafptr, cpuptr->addr, &cpuptr->data);
                cpuptr->step    = CPU_JMPIABS_5;
                break;
          case CPU_JMPIABS_5 :
                cpuptr->addr    = (cpuptr->addr & 0xff00) | ((cpuptr->addr + 1) & 0xff);
                (*cpuptr->rafptr).rafget(cpuptr->rafptr, cpuptr->addr, &data);
                cpuptr->pcreg   = cpuptr->data | (data << 8);
                cpuptr->step    = CPU_OPC;
//              taskloghex32(cpuptr->pcreg);
//              tasklogreturn();
                break;


        /* c65c02 timing! */
          case CPU_JSRABS_2 :
//              taskloghex32(cpuptr->pcreg);
//              tasklogstring(": JSR to address ");
                (*cpuptr->rafptr).rafget(cpuptr->rafptr, cpuptr->pcreg, &cpuptr->data);
                cpuptr->pcreg   = (cpuptr->pcreg + 1) & 0xffff;
                cpuptr->step    = CPU_JSRABS_3;
                break;
          case CPU_JSRABS_3 :
                (*cpuptr->rafptr).rafget(cpuptr->rafptr, cpuptr->spreg | 0x100, &data);
                cpuptr->step    = CPU_JSRABS_4;
                break;
          case CPU_JSRABS_4 :
                (*cpuptr->rafptr).rafset(cpuptr->rafptr, cpuptr->spreg | 0x100, cpuptr->pcreg >> 8);
                cpuptr->spreg--;
                cpuptr->step    = CPU_JSRABS_5;
                break;
          case CPU_JSRABS_5 :
                (*cpuptr->rafptr).rafset(cpuptr->rafptr, cpuptr->spreg | 0x100, cpuptr->pcreg & 0xff);
                cpuptr->spreg--;
                cpuptr->step    = CPU_JSRABS_6;
                break;
          case CPU_JSRABS_6 :
                (*cpuptr->rafptr).rafget(cpuptr->rafptr, cpuptr->pcreg, &data);
                cpuptr->pcreg   = cpuptr->data | (data << 8);
                cpuptr->step    = CPU_OPC;
//              taskloghex32(cpuptr->pcreg);
//              tasklogreturn();
                break;


          case CPU_RTS_2 :
//              taskloghex32(cpuptr->pcreg);
//              tasklogstring(": RTS to address ");
                (*cpuptr->rafptr).rafget(cpuptr->rafptr, cpuptr->pcreg, &data);
                cpuptr->step    = CPU_RTS_3;
                break;
          case CPU_RTS_3 :
                (*cpuptr->rafptr).rafget(cpuptr->rafptr, cpuptr->spreg | 0x100, &data);
                cpuptr->step    = CPU_RTS_4;
                break;
          case CPU_RTS_4 :
                cpuptr->spreg++;
                (*cpuptr->rafptr).rafget(cpuptr->rafptr, cpuptr->spreg | 0x100, &cpuptr->data);
                cpuptr->step    = CPU_RTS_5;
                break;
          case CPU_RTS_5 :
                cpuptr->spreg++;
                (*cpuptr->rafptr).rafget(cpuptr->rafptr, cpuptr->spreg | 0x100, &data);
                cpuptr->addr    = (cpuptr->data) | (data << 8);
                cpuptr->step    = CPU_RTS_6;
                break;
          case CPU_RTS_6 :
                cpuptr->pcreg   = (cpuptr->addr + 1) & 0xffff;
                (*cpuptr->rafptr).rafget(cpuptr->rafptr, cpuptr->spreg | 0x100, &data);
                cpuptr->step    = CPU_OPC;
//              taskloghex32(cpuptr->pcreg);
//              tasklogreturn();
                break;


          case CPU_RTI_2 :
//              taskloghex32(cpuptr->pcreg);
//              tasklogstring(": RTI to address ");
                (*cpuptr->rafptr).rafget(cpuptr->rafptr, cpuptr->pcreg, &data);
                cpuptr->step    = CPU_RTI_3;
                break;
          case CPU_RTI_3 :
                (*cpuptr->rafptr).rafget(cpuptr->rafptr, cpuptr->spreg | 0x100, &data);
                cpuptr->step    = CPU_RTI_4;
                break;
          case CPU_RTI_4 :
                cpuptr->spreg++;
                (*cpuptr->rafptr).rafget(cpuptr->rafptr, cpuptr->spreg | 0x100, &data);
                cpu6502setflags(cpuptr, data);
                cpuptr->step    = CPU_RTI_5;
                break;
          case CPU_RTI_5 :
                cpuptr->spreg++;
                (*cpuptr->rafptr).rafget(cpuptr->rafptr, cpuptr->spreg | 0x100, &cpuptr->data);
                cpuptr->step    = CPU_RTI_6;
                break;
          case CPU_RTI_6 :
                cpuptr->spreg++;
                (*cpuptr->rafptr).rafget(cpuptr->rafptr, cpuptr->spreg | 0x100, &data);
                cpuptr->pcreg   = (cpuptr->data) | (data << 8);
                cpuptr->step    = CPU_OPC;
//              taskloghex32(LOG_VERBOSE, cpuptr->pcreg);
//              tasklogreturn(LOG_VERBOSE);
                break;

          case CPU_COP_2 :
                cpuptr->pcreg           = (cpuptr->pcreg - 1) & 0xffff;
                cpuptr->stateflags      = cpuptr->stateflags | CPU_STATECOP;
                cpuptr->step            = CPU_OPC;
//              taskloghex32(LOG_VERBOSE, cpuptr->pcreg);
//              tasklogstring(LOG_VERBOSE, "| Coprocessor instruction reached");
//              tasklogreturn(LOG_VERBOSE);
                break;

          default :
                cpuptr->step            = CPU_OPC;
                cpuptr->stateflags      = cpuptr->stateflags | CPU_STATEGURU;
                taskloghex32(LOG_VERBOSE, cpuptr->pcreg);
                tasklogstring(LOG_VERBOSE, " | Undefined step cycle");
                return taskseterror("Undefined step cycle");

        } // switch (cpuptr->step)

        cpuptr->cycle++;

        return NULL;

      } // cpu6502step


// --> ifndef DEF_INC_CPU6502_C
#endif
