/*
 * EMU][ Apple ][-class emulator
 * Copyright (C) 2002, 2003 by the EMU][ Project/Dapple ][ Team
 *
 * Component:  MEMORY: routines for emulating 64KB Apple RAM and 3072KB
 *                     RAM Works expansion RAM (3136K total)
 *
 * This version dynamically allocates the RAM Works memory.  Not all memory
 * is recognized by some programs; what memory *is* recognized passes the
 * RAM Works diagnostic tests, and more than 2MB of this RAM is recognized
 * and usable by AppleWorks 3.0.  This emulation is based on an extract of
 * the RAM Works manual provided by Paul R. Santa-Maria which was revealed
 * to the programmer by Scott Alfter.  The author would like to express his
 * thanks for providing this information to him on the Usenet newsgroup
 * <comp.sys.apple2.programmer>. -uso.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, version 2.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 *  Current exception for ASMLIB.O linkage, if Z80.C is not used
 *
 * 20040428  New headers generated for v0.4 release
 *           Z80 core development
 *
 */


/*----------------------------------------


        memory.c


        see include file memory.h for definitions


----------------------------------------*/


#ifndef DEF_INC_MEMORY_C
#define DEF_INC_MEMORY_C


// **** include general definition files

#include <dir.h>
#include <dos.h>
#include <pc.h>
//#include <stdio.h>            // included by memory.h
#include <stdlib.h>             // used for malloc(), free()
//#include <string.h>           // included by memory.h
#include <unistd.h>
//#include "..\libs\general.h"  // included by memory.h
//#include "..\libs\asmlib.h"   // included by memory.h
//#include "dapple.h"           // included by memory.h


// **** include definition file of this source

#include "memory.h"


// **** include further Emu][ specific files

#include "cpu65c02.h"
#include "disk.h"
#include "gui.h"
#include "mouse.h"
#include "video.h"


#ifdef REECOP
// rom.c
void rompatch(unsigned char *filename);
#endif


// **** constants

#define MEMAUXRAM       0x10000
#define MEMSLOTROM      0x20000
#define MEMROM          0x21000
#define MEMINTCXROM     0x24000
#define MEMROM2         0x25000
// Maximum number of 64K auxiliary memory pages for the RAM Works card
// Manual says this (3072K) is the max
#define MEMMAXRAM       48


// **** variables

// flags that handle memory access
unsigned char memrompath[260];
unsigned char memram[0x28000];          /* main and aux memory + 32kb rom                       */
unsigned int  memnumbbanks;             /* number of allocated memory banks                     */
unsigned char *memramptr[MEMMAXRAM];    /* list of pointers to aux ram and ramworks ram         */
unsigned char *memauxbankptr;           /* pointer to aux ram or ramworks ram bank              */
unsigned char memkey;                   /* apple key                                            */
unsigned int  memjoyx;                  /* apple joystick x                                     */
unsigned int  memjoyy;                  /* apple joystick y                                     */
unsigned char memjoybutton0;            /* apple open button                                    */
unsigned char memjoybutton1;            /* apple close button                                   */
unsigned int  memjoycycle;              /* timer counter                                        */
unsigned char memauxread;               /* read motherboard ram or aux ram                      */
unsigned char memauxwrite;              /* write motherboard ram or aux ram                     */
unsigned char memstore80;               /* 80column store motherboard ram or aux ram            */
unsigned char memintcxrom;              /* read from slot rom $c100-$cfff or internal rom       */
unsigned char memslotc3;                /* read from slot rom $c300-$c3ff or internal rom       */
unsigned char memlcaux;                 /* read $0-$1ff + $d000-$ffff from main or aux memory   */
unsigned char memlcramr;                /* read LC RAM or ROM?                                  */
unsigned char memlcramw;                /* write LC RAM or ROM?                                 */
unsigned char memlcbank2;               /* read from LC bank 1 or LC bank 2                     */
unsigned char memauxbank;               /* which bank selected?                                 */
unsigned char memiic32k;                /* //c rom 16k or 32k                                   */
IIC4080       memiic4080;               /* 40/80 column mode switch                             */
unsigned char memrombank;               /* for Apple//c only                                    */
unsigned char memann0;                  /* annunciator #0                                       */
unsigned char memann1;                  /* annunciator #1                                       */
unsigned char memann2;                  /* annunciator #2                                       */
unsigned char memann3;                  /* annunciator #3                                       */
unsigned char memnolc;                  /* Is there a LC card installed?                        */
unsigned char memclearmode;             /* memory clear pattern                                 */
void memorywrite(unsigned int address, unsigned int value);

// flags for sound
unsigned char soundflag;
int           soundportval;






/*--------------------------------------*/
/* Sound                                */
/*--------------------------------------*/


      void soundinit(void) {
//      register int al;

        soundflag = 1;
        soundportval = inportb(0x61) & 0xfc;
//      al = al & 0xfc;
//      outportb(0x61, al);

      } // soundinit


/*--------------------------------------*/


      void soundclick(void) {
//      register int al;

        if (soundflag) {
        // Toggle the speaker
          soundportval = soundportval ^ 0x2;
          outportb(0x61, soundportval);
//        al = inportb(0x61);
//        al = al ^ 0x02;
//        al = al & 0xfe;
//        outportb(0x61,al);
        }

      } // soundclick


/*--------------------------------------*/


      void soundmenu(void) {
        unsigned int  menukeyboard;
        unsigned int  menuwindow;
        unsigned char key;
        unsigned char update;

        if (!windowaddio( -1, -1, WINDOWXSIZE, WINDOWYSIZE, -1, 1,
"!\
ESound Options;\
GSound Optionen;\
", 0, &menukeyboard, &menuwindow)) {
          update = 1;
          do {
            if (update) {
              channelout(menuwindow, 12);               /* clear window */
              stringwritemessage(menuwindow,
"!\
E\r[ESC] - Quit\r\r\r[S] - PC-Speaker: ;\
G\r[ESC] - Verlasse Men\r\r\r\[S] - PC-Speaker: ;\
");

              if (soundflag) {
                stringwritemessage(menuwindow,
"!\
Eenabled;\
Geingeschaltet;\
");
              }
              else {
                stringwritemessage(menuwindow,
"!\
Edisabled;\
Gausgeschaltet;\
");
              }
              update = 0;
              screenupdate = 1;
            }

            do {
              taskswitch();
              if (windowgetclose(menuwindow)) {
                key = 27;
              }
              else {
                key = (unsigned char)channelin(menukeyboard);
              }
            }
            while ((key == 0) && (!exitprogram));

            switch (key) {
              case 's' :
              case 'S' :
                soundflag = !soundflag;
                update = 1;
                break;
            } // switch (key)

          }
          while ((key != 27) && (key != 32) && (key != 13) && (!exitprogram));
          channelclose(menukeyboard);
          channelclose(menuwindow);
        }

      } // soundmenu


/*--------------------------------------*/
/* Memory                               */
/*--------------------------------------*/


      void memoryinit(void) {

        memramptr[0]    = &memram[MEMAUXRAM];           // set first pointer to standard aux ram
        for (memnumbbanks=1; memnumbbanks<MEMMAXRAM; memnumbbanks++) {  // clear pointer array
          memramptr[memnumbbanks] = NULL;
        }

        memnumbbanks = 0;                               // allocate memory
        do {
          memnumbbanks++;
          memramptr[memnumbbanks] = malloc(65536);
        }
        while ((memnumbbanks < (MEMMAXRAM-1)) && (memramptr[memnumbbanks] != NULL));

        memorysetclearmode(1);
        memoryclear();
        memnolc = 0;
        memjoyx         = 0x575;
        memjoyy         = 0x575;
        memjoybutton0   = 0;
        memjoybutton1   = 0;
        memiic4080      = IIC40;
        memiic32k       = 0;
        appletype       = 0;                            /* no roms loaded */
        memoryreset();
        strcpy(memrompath, inirompath);
        memoryrominit();

      } // memoryinit


/*--------------------------------------*/


      void memoryclose(void) {
        register unsigned int i;

        for (i=1; i < MEMMAXRAM; i++) {         /* free allocated memory */
          if (memramptr[i] != NULL) {
            free(memramptr[i]);
          }
        }

      } // memoryclose


/*--------------------------------------*/


      unsigned char *memoryrominit(void) {
#ifdef REECOP
        FILE *file;
        unsigned char filename[260];
#endif

        memset(&memram[MEMSLOTROM], 0x00, 0x8000);      // clear roms
#ifdef REECOP
        strcpy(filename, inirompath);
        strcat(filename, "\\NOROM.A65");
        file=fopen(filename, "rb");
        if (file) {
          fclose(file);
          rompatch(filename);
          appletype = APPLEII;
          virtsetcharset(virtgetcharset());             // adjust character rom
          return NULL;
        }
        return taskseterror("REE rom could not be loaded");
#else
        return NULL;
#endif

      } // memoryrominit


/*--------------------------------------*/


      unsigned int memoryloadrom (unsigned int keyboard, unsigned int window, unsigned char *newrompath) {
        FILE *file;
        unsigned char oldcwd[260];
        unsigned int  filelength;
        unsigned char guessedtype;
        register unsigned int i;

//      return 0;
        if (!*newrompath) return 0;

        stringwritemessage(window,
"!\
EAttempting to load Apple Rom: ';\
GVersuche, Apple Rom zu laden: ';\
");

        stringwrite(window, newrompath);
        stringwrite(window, "'\r");

        screenupdate = 1;
        taskswitch();

        cwdxchgslash(oldcwd, 256);
        chdir(inirompath);                      // restore rom directory in case only a file is given

        file=fopen(newrompath,"rb");
        if (!file) {
          stringwritemessage(window,
"!\
ESorry, I can't seem to open the rom file.\r;\
GDie ROM-Datei konnte nicht geffnet werden.\r;\
");
          windowpresskey(keyboard, window);
          chdir(oldcwd);                        // restore original directory
          return 0;
        }

        fseek(file,0,SEEK_END);
        filelength=ftell(file);
        fseek(file,0,SEEK_SET);
        stringwritemessage(window,
"!\
EReading rom file...\r;\
GLese ROM-Datei...\r;\
");
        screenupdate = 1;
        taskswitch();

        if (strright(newrompath, "Laser128.ROM")) {
          guessedtype = LASER128;
        }
        else {
          if (strright(newrompath, "e.ROM")
           || strright(newrompath, "eo.ROM")
           || strright(newrompath, "eu.ROM")) {
            guessedtype = APPLEIIE;
          }
          else {
            if (strright(newrompath, "c.ROM")
             || strright(newrompath, "co.ROM")) {
              guessedtype = APPLEIIC;
            }
            else {
              guessedtype = APPLEII;
            }
          }
        }

        switch (filelength) {
          case 12288 :  /* AppleII+ */
            memset(&memram[0x20000], 0x00, 0x8000);
            fread(&memram[MEMROM],      12288, 1, file);                /* read rom */
            appletype = APPLEII;                                        /* machine is Apple][+ */
            break;
          case 16128 :  /* 15.75KB = AppleIIe / Apple//c original */
            memset(&memram[0x20000], 0x00, 0x8000);
            if (guessedtype == APPLEIIC) {                              /* 16K //c ROM (!) */
              fread(&memram[MEMSLOTROM+0x100], 3840, 1, file);          /* read slot rom */
//            slotroms = 1;                                             /* found slot roms */
              fread(&memram[MEMROM],12288,1,file);                      /* read rom */
              for (i=0x0000; i<0x1000; i++) {                           /* make slot rom = intcxrom */
                memram[i+MEMINTCXROM]   = memram[i+MEMSLOTROM];
              }
              for (i=0x0000; i<0x3000; i++) {                           /* copy rom */
                memram[i+MEMROM2]       = memram[i+MEMROM];
              }
              appletype = APPLEIIC;
              memiic32k=0;                                              /* old 16 KB rom */
            }
            else {      /* AppleIIe */
              fread(&memram[MEMINTCXROM+0x100], 3840, 1, file);         /* read INTCXROM */
              fread(&memram[MEMROM],12288,1,file);                      /* read rom */
              appletype = APPLEIIE;                                     /* machine is Apple//e */
            }
            break;
          case 16384 :  /* 16KB = AppleII+ with slots / AppleIIe with INTCXROM */
            memset(&memram[0x20000], 0x00, 0x8000);
            if (guessedtype == APPLEIIE) {
              fread(&memram[MEMINTCXROM],4096,  1, file);               /* read intcxrom */
              fread(&memram[MEMROM],    12288,  1, file);               /* read rom */
              appletype = APPLEIIE;                                     /* machine is Apple//e */
            }
            else {
              fread(&memram[MEMROM],    12288,  1, file);               /* read rom */
              fread(&memram[MEMSLOTROM],  4096, 1, file);               /* read slot rom */
              appletype = APPLEII;                                      /* machine is Apple][+ */
            }
            break;
          case 20480 :  /* 20KB = Apple II+ (ApplePC) / Apple//e (AppleWin) */
            memset(&memram[0x20000], 0x00, 0x8000);
            if (guessedtype == APPLEIIE) {
              fread(&memram[MEMSLOTROM], 4096,  1, file);               /* read slot rom */
              fread(&memram[MEMINTCXROM],4096,  1, file);               /* read intcxrom */
              fread(&memram[MEMROM],    12288,  1, file);               /* read rom */
              appletype = APPLEIIE;                                     /* machine is Apple//e */
            }
            else {
              fseek(file,4096,SEEK_SET);                                /* skip first 4096 bytes) */
              fread(&memram[MEMSLOTROM], 4096,  1, file);               /* read slot rom */
              fread(&memram[MEMROM],    12288,  1, file);               /* read rom */
              appletype = APPLEII;                                      /* machine is Apple][+ */
            }
            break;
          case 32768 :
            memset(&memram[0x20000], 0x00, 0x8000);
            if (guessedtype == APPLEIIC) {
              fread(&memram[MEMSLOTROM], 4096,  1, file);               /* read slot rom */
              fread(&memram[MEMROM],    12288,  1, file);               /* read rom */
              fread(&memram[MEMINTCXROM],4096,  1, file);               /* read intcxrom */
              fread(&memram[MEMROM2],   12288,  1, file);               /* read rom bank 2 */
              appletype = APPLEIIC;                                     /* machine is Apple//c */
              memiic32k = 1;                                            /* new 32KB rom */
            }
            else {
              if (guessedtype == LASER128) {
                fread(&memram[MEMINTCXROM],4096,        1, file);       /* read slot rom */
                fread(&memram[MEMROM],    12288,        1, file);       /* read rom */
                fread(&memram[MEMSLOTROM], 4096,        1, file);       /* read intcxrom */
                fread(&memram[MEMROM2],   12288,        1, file);       /* read rom bank 2 */
                appletype = LASER128;                                   /* machine is Laser128 */
              }
              else {
                fread(&memram[MEMSLOTROM], 4096,        1, file);       /* read slot rom */
                fread(&memram[MEMROM],    12288,        1, file);       /* read rom */
                fread(&memram[MEMINTCXROM],4096,        1, file);       /* read intcxrom */
                appletype = APPLEIIE;                                   /* machine is Apple//e */
              }
            }
            break;
          default :
            fclose(file);
            stringwritemessage(window,
"!\
EThe file does not contain a known type of rom.\r;\
GDie Datei enthlt kein bekanntes AppleII ROM.\r;\
");
            windowpresskey(keyboard, window);
            chdir(oldcwd);                      /* restore original directory */
            return 0;
        } /* switch */
        fclose(file);
        if (appletype == APPLEII) {
          stringwritemessage(window,
"!\
EMachine is Apple][ or Apple][+.\r;\
GComputer ist Apple][ oder Apple][+.\r;\
");

#ifdef REECOP
          stringwritemessage(window,
"!\
EPatching rom...\r;\
GPatche rom...\r;\
");
          rompatch("ROMPATCH.A65");
#endif
        }
        else {
          if (appletype == APPLEIIE) {
            stringwritemessage(window,
"!\
EMachine is Apple//e.\r;\
GComputer ist Apple//e.\r;\
");
          }
          else {
            if (appletype == LASER128) {
              stringwritemessage(window,
"!\
EMachine is Laser 128.\r;\
GComputer ist Laser 128.\r;\
");
            }
            else {
              if (memiic32k) {
                stringwritemessage(window,
"!\
EMachine is Apple//c.\r;\
GComputer ist Apple//c.\r;\
");
              }
              else {
                stringwritemessage(window,
"!\
EMachine is Apple//c (16K ROM).\r;\
GComputer ist Apple//c (16K ROM).\r;\
");
              }
            }
          }
        }
        screenupdate = 1;
        taskswitch();

// init memory

        strcpy(memrompath, newrompath);
        getpathstring(inirompath, newrompath);  // new rom path
        chdir(oldcwd);                          // restore original directory

        stringwritemessage(window,
"!\
EResetting Apple...\r;\
GReset Apple...\r;\
");
        screenupdate = 1;
        taskswitch();

        memoryclear();
        virtsetcharset(virtgetcharset());       // adjust character rom
        applereset();

        return -1;
      } // memoryloadrom


/*--------------------------------------*/


// automatic search for rom file
      unsigned int memoryautoloadrom(unsigned int keyboard, unsigned int window) {
        FILE *rom;
        unsigned int i;

/* Try Apple//e first, then Apple//c, then Apple][+ */

        rom = fopen(memrompath,"rb");
        i = (unsigned int)(rom);
        if (i) {
          fclose(rom);
          i = memoryloadrom(keyboard, window, memrompath);
        }
        if (!i) {
          strcpy(memrompath, inirompath);
          strcat(memrompath,"\\apple2e.rom");
          rom = fopen(memrompath,"rb");
          i = (unsigned int)(rom);
          if (i) {
            fclose(rom);
            i = memoryloadrom(keyboard, window, memrompath);
          }
          if (!i) {
            strcpy(memrompath, inirompath);
            strcat(memrompath,"\\apple2eo.rom");
            rom = fopen(memrompath,"rb");
            i = (unsigned int)(rom);
            if (i) {
              fclose(rom);
              i = memoryloadrom(keyboard, window, memrompath);
            }
            if (!i) {
              strcpy(memrompath, inirompath);
              strcat(memrompath,"\\apple2e.rom");
              rom = fopen(memrompath,"rb");
              i = (unsigned int)(rom);
              if (i) {
                fclose(rom);
                i = memoryloadrom(keyboard, window, memrompath);
              }
              if (!i) {
                strcpy(memrompath, inirompath);
                strcat(memrompath,"\\apple2co.rom");
                rom = fopen(memrompath,"rb");
                i = (unsigned int)(rom);
                if (i) {
                  fclose(rom);
                  i = memoryloadrom(keyboard, window, memrompath);
                }
                if (!i) {
                  strcpy(memrompath, inirompath);
                  strcat(memrompath,"\\apple2c.rom");
                  rom = fopen(memrompath,"rb");
                  i = (unsigned int)(rom);
                  if (i) {
                    fclose(rom);
                    i = memoryloadrom(keyboard, window, memrompath);
                  }
                  if (!i) {
                    strcpy(memrompath, inirompath);
                    strcat(memrompath,"\\apple2.rom");
                    rom = fopen(memrompath,"rb");
                    i = (unsigned int)(rom);
                    if (i) {
                      fclose(rom);
                      i = memoryloadrom(keyboard, window, memrompath);
                    }
                    if (!i) {
                      strcpy(memrompath, inirompath);
                      strcat(memrompath,"\\apple2m.rom");
                      rom = fopen(memrompath,"rb");
                      i = (unsigned int)(rom);
                      if (i) {
                        fclose(rom);
                        i = memoryloadrom(keyboard, window, memrompath);
                      }
                      if (!i) {
                        strcpy(memrompath, inirompath);
                        strcat(memrompath,"\\apple2o.rom");
                        rom = fopen(memrompath,"rb");
                        i = (unsigned int)(rom);
                        if (i) {
                          fclose(rom);
                          i = memoryloadrom(keyboard, window, memrompath);
                        }
                        if (!i) {
                          strcpy(memrompath, inirompath);
                          strcat(memrompath,"\\laser128.rom");
                          rom = fopen(memrompath,"rb");
                          i = (unsigned int)(rom);
                          if (i) {
                            fclose(rom);
                            i = memoryloadrom(keyboard, window, memrompath);
                          }
                          if (!i) {
                            if (memoryrominit()) {
                              strcpy(memrompath, "");
                              stringwrite(window, "No Apple ROMS found\r");
                              windowpresskey(keyboard, window);
                              return 0;
                            }
                          }
                        }
                      }
                    }
                  }
                }
              }
            }
          }
        }
        return -1;
      } // memoryautoloadrom


/*--------------------------------------*/


      void memoryreset(void) {

        memkey          = memkey & 0x7f;
        memauxread      = 0;
        memauxwrite     = 0;
        memintcxrom     = 0;            /* read from slot rom $c100-$cfff               */
        memslotc3       = 0;            /* read slot rom                                */
        memlcaux        = 0;            /* read $0-$1ff + $d000-$ffff from main memory  */
        memlcramr       = 0;            /* read ROM                                     */
        memlcramw       = 0;            /* write ROM                                    */
        memlcbank2      = 0;            /* read from LC bank 1                          */
        memrombank      = 0;            /* read from ROM bank 1                         */
        memstore80      = 0;            /* 80 store off                                 */
        memann0         = 0x80;         /* annunciator #0 on                            */
        memann1         = 0x80;         /* annunciator #1 on                            */
        memann2         = 0x80;         /* annunciator #2 on                            */
        memann3         = 0x80;         /* annunciator #3 on                            */

        memjoycycle     = cpugetcycle();

      } // memoryreset


/*-------------------------------------*/


      void memoryresetmenu(void) {
        unsigned int menukeyboard;
        unsigned int menuwindow;
        unsigned char message[60];


        if (windowaddio( -1, -1, 400, 56, -1, 1,
"!\
EReset Computer;\
GReset Computer;\
", 0, &menukeyboard, &menuwindow)) {
          exitprogram = 1;
          return;
        }
        channelout(menuwindow, 13);
        stringcopymessage(message,
"!\
EReset computer: Are you sure?;\
GReset Computer: Sind Sie sicher?;\
", 59);
        stringwritecenterx(menuwindow, message);
        if (guiyesno(menukeyboard, menuwindow)) {
          channelclose(menukeyboard);
          channelclose(menuwindow);
          applereset();
        }
        else {
          channelclose(menukeyboard);
          channelclose(menuwindow);
        }
      } // memoryresetmenu


/*--------------------------------------*/


      void memoryclearbank(unsigned char membank[65536]) {
        register unsigned int i;
        register unsigned int j;

        switch (memclearmode) {
          case 0 :                      // None
            break;
          case 1 :                      // 00 00 00 00
            memset(membank, 0x00, 0x10000);
            break;
          case 2 :                      // 00 FF 00 FF
            for (i=0; i<0x10000; ) {
              membank[i++] = 0x00;
              membank[i++] = 0xff;
            }
            break;
          case 3 :                      // 00 FF .. FF 00
            for (i=0; i<0x10000; ) {
              for (j=0; j<0x80; j++) {
                membank[i++] = 0x00;
                membank[i++] = 0xff;
              }
              for (j=0; j<0x80; j++) {
                membank[i++] = 0xff;
                membank[i++] = 0x00;
              }
            }
            break;
          case 4 :                      // EA EA EA EA
            memset(membank, 0xea, 0x10000);
            break;
          case 5 :
            for (i=0; i<0x10000; i++) {
              membank[i] = (i & 0xff);
            }
            break;
          case 6 :                      // Dead Beef Little Endian
            for (i=0; i<0x10000; i=i+4) {
              membank[i+3]=0xDE;
              membank[i+2]=0xAD;
              membank[i+1]=0xBE;
              membank[i]  =0xEF;
           } // for (i)
           break;
          case 7 :                      // Dead Beef High Endian
            for (i=0; i<0x10000; i=i+4) {
              membank[i]  =0xDE;
              membank[i+1]=0xAD;
              membank[i+2]=0xBE;
              membank[i+3]=0xEF;
           } // for (i)
           break;
        } // switch (memclearmode)

      } // memoryclearbank


      void memoryclear(void) {
        register unsigned int i;

        memoryclearbank(&memram[0]);            /* clear main memory */
        for (i=0; i<MEMMAXRAM; i++) {           /* clear aux and ramworks memory */
          if (memramptr[i] != NULL) {
            memoryclearbank(memramptr[i]);
          }
        }

        memauxbank = 0;                         /* reset the RAM Works */
        memauxbankptr = &memram[MEMAUXRAM];

        memkey = 0;

      } // memoryclear


/*--------------------------------------*/


      void memorysetclearmode(unsigned int value) {

        if (value <= 7) {
          memclearmode = value;
        }

      } // memorysetclearmode


/*--------------------------------------*/


      unsigned int memorygetclearmode(void) {

        return memclearmode;

      } // memorygetclearmode


/*-------------------------------------*/


      void memoryclearresetmenu(void) {
        unsigned int menukeyboard;
        unsigned int menuwindow;
        unsigned char message[60];


        if (windowaddio( -1, -1, 480, 56, -1, 1,
"!\
EClear Memory And Reset Computer;\
GSpeicher lschen und Reset;\
", 0, &menukeyboard, &menuwindow)) {
          exitprogram = 1;
          return;
        }
        channelout(menuwindow, 13);
        stringcopymessage(message,
"!\
EClear memory and reset computer: Are you sure?;\
GLsche Speicher und reset Computer: Sind Sie sicher?;\
", 59);
        message[59] = 0;
        stringwritecenterx(menuwindow, message);

        if (guiyesno(menukeyboard, menuwindow)) {
          channelclose(menukeyboard);
          channelclose(menuwindow);
          memoryclear();
          applereset();
        }
        else {
          channelclose(menukeyboard);
          channelclose(menuwindow);
        }
      } // memoryclearresetmenu


/*--------------------------------------*/


      static int match (char *s1, char *s2) {
        if (s1[0]!=s2[0]) return -1;
        if (s1[1]!=s2[1]) return -1;
        if (s1[2]!=s2[2]) return -1;
        if (s1[3]!=s2[3]) return -1;
        return 0;
      } // match


      int memorybloadfile(unsigned char *filename) {
        FILE *file;
        unsigned int keyboard;
        unsigned int window;
        int error;
        unsigned char savedebugflag;
        unsigned char uid[4];
        unsigned int addr;
        unsigned int length;
        unsigned int saveaddr;

        error = 0;
        if (!(windowprotocol(&keyboard, &window,
"!\
EBload File;\
GBload Datei;\
"))) {
          stringwritemessage(window,
"!\
EAttempting to bload file:\r\';\
GVersuche Datei zu 'bloaden':\r\';\
");
          stringwrite(window, filename);
          guipercent(window, 0, "\'\r");

          file=fopen (filename,"rb");
          if (!file) {
            stringwritemessage(window,
"!\
EError opening file.\r;\
GFehler beim ffen der Datei.\r;\
");
            stringwrite(window, stringgeterror());      /* display error */
            channelout(window, 13);
            error = 2;
          }
          else {
            fread(&uid, sizeof(uid), 1, file);          /* read first 4 bytes */
            addr = (0xde << 24) | (0xad << 16) | (0xbe << 8) | (0xef); /* $DEADBEEF Big Endian */
            if (!match(uid, (unsigned char *)&addr)) {
              fread(uid,4,1,file);
              if (match(uid,"DPL2")) {
                stringwrite(window, "Header code 'DPL2' not found.\r");
                error = 3;      /* DPL2 header not found */
              }
              else {
                fread(uid,4,1,file);
                if (match(uid,"BINA")) {
                  stringwrite(window, "Header code 'BINA' not found.\r");
                  error = 4;    /* type not found */
                }
                else {
                  fread(&uid, 4, 1, file);      /* read start address and length */
                }
              }
            }
            if (!error) {
              addr   = (uid[1] << 8) | uid[0];  /* make code transferable to other platforms */
              length = (uid[3] << 8) | uid[2];
              saveaddr = addr;

              guipercent(window, 50,
"!\
EReading data from file...;\
GLese Daten aus Datei...;\
");
              screenupdate = 1;
              taskswitch();
              savedebugflag = inidebugflag;     /* save flag in case we are bloading from a (forthcoming) debugger */
              inidebugflag = 1;                 /* set this flag if you don't want writes to the softswitch area */
              for (; length; length--) {
                fread(&uid[0], 1, 1, file);
                memorywrite(addr, uid[0]);
                addr = (addr +  1) & 0xffff;    /* do wrap around after address 0xffff */
              } /* for */
              inidebugflag = savedebugflag;
              guipercent(window, 100,
"!\
EData read.;\
GDaten gelesen.;\
");
              stringwritemessage(window,
"!\
EDo you want to start the program?;\
GMchten Sie das Programm starten?;\
");
              if (guiyesno(keyboard, window)) {
                cpusetpc(saveaddr);
                cpusetsp(0xff);
              }
            }
            fclose(file);
          }

          if (error) {
            windowpresskey(keyboard, window);
          }
          channelclose(keyboard);
          channelclose(window);
        }
        else {
          error = 1;    // could not open window
        }
        return error;
      } // memorybloadfile


/*--------------------------------------*/


      unsigned char *memorystore(unsigned int winprotocol, FILE *file, unsigned int percent) {
        unsigned char header[32];

        memset(header, 0x00, sizeof(header));
        strcpy(header, "MEMORY STATE V0.27");
        fwrite(header,           sizeof(header),        1, file);

        fwrite(&soundflag,      sizeof(soundflag),      1,file);
        fwrite(&appletype,      sizeof(appletype),      1,file);
        fwrite(inirompath,      sizeof(inirompath),     1,file);
        fwrite(memrompath,      sizeof(memrompath),     1,file);
        fwrite(&memnolc,        sizeof(memnolc),        1,file);
        fwrite(&memram[0],      49152,                  1,file);
        if ((!memnolc) || (!(appletype & APPLEII))) {
          fwrite(&memram[49152],16384,                  1,file);
        }
        if (appletype & (APPLEIIE | APPLEIIC)) {
          fwrite(&memram[65536],49152,                  1,file);
          if ((!memnolc) || (!(appletype & APPLEII))) {
            fwrite(&memram[65536+49152],16384,          1,file);
          }
        }
        fwrite(&memram[0x20000],32768,                  1,file);
        fwrite(&memauxbank,     sizeof(memauxbank),     1,file);
        fwrite(&memauxread,     sizeof(memauxread),     1,file);
        fwrite(&memauxwrite,    sizeof(memauxwrite),    1,file);
        fwrite(&memintcxrom,    sizeof(memintcxrom),    1,file);
        fwrite(&memslotc3,      sizeof(memslotc3),      1,file);
        fwrite(&memlcaux,       sizeof(memlcaux),       1,file);
        fwrite(&memlcramr,      sizeof(memlcramr),      1,file);
        fwrite(&memlcramw,      sizeof(memlcramw),      1,file);
        fwrite(&memlcbank2,     sizeof(memlcbank2),     1,file);
        fwrite(&memstore80,     sizeof(memstore80),     1,file);
        fwrite(&memrombank,     sizeof(memrombank),     1,file);
        fwrite(&memiic32k,      sizeof(memiic32k),      1,file);
        fwrite(&memiic4080,     sizeof(memiic4080),     1,file);
        fwrite(&memann0,        sizeof(memann0),        1,file);
        fwrite(&memann1,        sizeof(memann1),        1,file);
        fwrite(&memann2,        sizeof(memann2),        1,file);
        fwrite(&memann3,        sizeof(memann3),        1,file);
        fwrite(&memkey,         sizeof(memkey),         1,file);
        fwrite(&memjoyx,        sizeof(memjoyx),        1,file);
        fwrite(&memjoyy,        sizeof(memjoyy),        1,file);
        fwrite(&memjoybutton0,  sizeof(memjoybutton0),  1,file);
        fwrite(&memjoybutton1,  sizeof(memjoybutton1),  1,file);
        fwrite(&memjoycycle,    sizeof(memjoycycle),    1,file);

        guipercent(winprotocol, percent,
"!\
EMemory stored.;\
GArbeitsspeicher gespeichert.;\
");

        return NULL;            // no error

      } // memorystore


/*--------------------------------------*/


      unsigned char *memoryrestore(unsigned int winprotocol, FILE *file, unsigned int percent) {
        unsigned char header[32];

        fread(header,           sizeof(header),         1,file);
        if (strcmp(header, "MEMORY STATE V0.27")) {
          stringwrite(winprotocol, "Memory emulation data not found.\r");
          return taskseterror("Memory emulation data not found.");
        }

        fread(&soundflag,       sizeof(soundflag),      1,file);
        fread(&appletype,       sizeof(appletype),      1,file);
        fread(inirompath,       sizeof(inirompath),     1,file);
        fread(memrompath,       sizeof(memrompath),     1,file);
        fread(&memnolc,         sizeof(memnolc),        1,file);
        fread(&memram[0],       49152,                  1,file);
        if ((!memnolc) || (!(appletype & APPLEII))) {
          fread(&memram[49152], 16384,                  1,file);
        }
        if (appletype & (APPLEIIE | APPLEIIC)) {
          fread(&memram[65536], 49152,                  1,file);
          if ((!memnolc) || (!(appletype & APPLEII))) {
            fread(&memram[65536+49152],16384,           1,file);
          }
        }
        fread(&memram[0x20000], 32768,                  1,file);
        fread(&memauxbank,      sizeof(memauxbank),     1,file);
        fread(&memauxread,      sizeof(memauxread),     1,file);
        fread(&memauxwrite,     sizeof(memauxwrite),    1,file);
        fread(&memintcxrom,     sizeof(memintcxrom),    1,file);
        fread(&memslotc3,       sizeof(memslotc3),      1,file);
        fread(&memlcaux,        sizeof(memlcaux),       1,file);
        fread(&memlcramr,       sizeof(memlcramr),      1,file);
        fread(&memlcramw,       sizeof(memlcramw),      1,file);
        fread(&memlcbank2,      sizeof(memlcbank2),     1,file);
        fread(&memstore80,      sizeof(memstore80),     1,file);
        fread(&memrombank,      sizeof(memrombank),     1,file);
        fread(&memiic32k,       sizeof(memiic32k),      1,file);
        fread(&memiic4080,      sizeof(memiic4080),     1,file);
        fread(&memann0,         sizeof(memann0),        1,file);
        fread(&memann1,         sizeof(memann1),        1,file);
        fread(&memann2,         sizeof(memann2),        1,file);
        fread(&memann3,         sizeof(memann3),        1,file);
        fread(&memkey,          sizeof(memkey),         1,file);
        fread(&memjoyx,         sizeof(memjoyx),        1,file);
        fread(&memjoyy,         sizeof(memjoyy),        1,file);
        fread(&memjoybutton0,   sizeof(memjoybutton0),  1,file);
        fread(&memjoybutton1,   sizeof(memjoybutton1),  1,file);
        fread(&memjoycycle,     sizeof(memjoycycle),    1,file);

        guipercent(winprotocol, percent,
"!\
EMemory restored.\
GArbeitsspeicher geladen.\
");

        return NULL;            // no error

      } // memoryrestore


/*--------------------------------------*/


      unsigned char memoryread(unsigned int addr) {

        if (addr < 0xc000) {
          if (addr < 0x2000) {
            if (addr < 0x400) {
              if (addr < 0x200) {
/* $0000 - $01ff */
                if (memlcaux)           { return memauxbankptr[addr]; }         /* aux 64 */
                else                    { return memram[addr]; }                /* main 64 */
              }
              else {
/* $0200 - $03ff */
                if (memauxread)         { return memauxbankptr[addr]; }         /* aux 64 */
                else                    { return memram[addr]; }                /* main 64 */
              }
            }
            else {
              if (addr < 0x800) {
/* $0400 - $07ff */
                if (memstore80) {                       /* not affected by RAM Works -uso. */
                  if (virtpage2)        { return memram[addr+MEMAUXRAM]; }      /* aux 64 */
                  else                  { return memram[addr]; }                /* main 64 */
                }
                else {
                  if (memauxread)       { return memauxbankptr[addr]; }         /* aux 64 */
                  else                  { return memram[addr]; }                /* main 64 */
                 }
               }
               else {
/* $0800 - $1fff */
                if (memauxread)         { return memauxbankptr[addr]; }         /* aux 64 */
                else                    { return memram[addr]; }                /* main 64 */
              }
            }
          } /* if (addr < 0x2000) */
          else {
            if (addr < 0x4000) {
/* $2000 - $3fff */
              if ((memstore80) && (virthgr)) {          /* not affected by RAM Works -uso. */
                if (virtpage2)          { return memram[addr+MEMAUXRAM]; }      /* aux 64 */
                else                    { return memram[addr]; }                /* main 64 */
              }
              else {
                if (memauxread)         { return memauxbankptr[addr]; }         /* aux 64 */
                else                    { return memram[addr]; }                /* main 64 */
              }
            }
            else {
/* $4000 - $bfff */
              if (memauxread)           { return memauxbankptr[addr]; }         /* aux 64 */
              else                      { return memram[addr]; }                /* main 64 */
            }
          }
        } /* if (addr < 0xc000) */
        else {
          if (addr >= 0xd000) {
            if ((!memlcramr) || (memnolc && (appletype & APPLEII)) ) {
              if (memrombank) {
                return memram[addr-0xd000+MEMROM2];     /* ROM bank 2 */
              }
              else {
                return memram[addr-0xd000+MEMROM];      /* ROM is the same for whatever memlcaux is set to */
              }
            }
            else {
              if (memlcaux) {
                if (addr < 0xe000) {
                  if (!memlcbank2) {
                    return memauxbankptr[(addr-0x1000)];
                  }
                }
                return memauxbankptr[addr];
              }
              else {
                if (addr < 0xe000) {
                  if (!memlcbank2) {
                    return memram[addr-0x1000];
                  }
                }
                return memram[addr];
              }
            } /* else if (memlcramr) */
          } /* if (addr => 0xd000 */
          else {

 /* test for softswitch area */
            if (addr <= 0xc0ff) {

              if (addr <= 0xc00f) { return memkey; }    /* keyboard */

              if (addr <= 0xc01f) {
                if (appletype & (APPLEIIE | APPLEIIC)) {        /* not on Apple][+ */
                  switch (addr) {
                    case 0xc010 : memkey = memkey & 0x7f;
                                  return memkey;        /* not correct (undefined so far) */
                    case 0xc011 : return (memlcbank2    | (memkey&0x7f));
                    case 0xc012 : return (memlcramr     | (memkey&0x7f));
                    case 0xc013 : return (memauxread    | (memkey&0x7f));
                    case 0xc014 : return (memauxwrite   | (memkey&0x7f));
                    case 0xc015 : if (appletype & APPLEIIE) {
                                    return (memintcxrom | (memkey&0x7f));
                                  }
                                  else {
                                    return memkey&0x7f;
                                  }
                    case 0xc016 : return (memlcaux      | (memkey&0x7f));
                    case 0xc017 : if (appletype & APPLEIIE) {
                                    return (memslotc3   | (memkey&0x7f));
                                  }
                                  else {
                                    return memkey&0x7f;
                                  }
                    case 0xc018 : return (memstore80    | (memkey&0x7f));
                    case 0xc019 : if (appletype & APPLEIIE) {
                                    return (virtrasterline<192) ? ((memkey&0x7f)|0x80) : (memkey&0x7f);
                                  }
                                  else {
                                    return memkey&0x7f;
                                  }
                    case 0xc01a : return ((virtgraphic ^ 0x80) | (memkey&0x7f));
                    case 0xc01b : return (virtmixed     | (memkey&0x7f));
                    case 0xc01c : return (virtpage2     | (memkey&0x7f));
                    case 0xc01d : return (virthgr       | (memkey&0x7f));
                    case 0xc01e : return (virtaltchar   | (memkey&0x7f));
                    case 0xc01f : return (virt80col     | (memkey&0x7f));
                  } /* switch */
                }
                else {
                  memkey = memkey & 0x7f;
                  return memkey;        /* not correct (undefined so far) */
                }
              }


              if (addr <= 0xc02f) {
                if (appletype & APPLEIIC) {
                  if (addr == 0xc028) {
                    if (!inidebugflag) {
                      memrombank = !memrombank;
                    }
                  }
                  return virtvideobyte&0x7f;
                }
                else {
                  return virtvideobyte&0x7f;            /* Cassette output toggle */
                }
              }

              if (addr <= 0xc03f) {
                if (!inidebugflag) {
                  soundclick();
                }
                return virtvideobyte&0x7f;              /* Speaker */
              }

              if (addr <= 0xc04f) {
                return virtvideobyte&0x7f;              /* Utility output strobe - nani yo? */
              }

              if (inidebugflag) {
                return virtvideobyte;
              }

              if (addr <= 0xc08f) {
                switch (addr) {
                  case 0xc050 : if (!virtgraphic) {
                                  virtgraphic = 0x80;
                                  virtsetmode();
                                }
                                return virtvideobyte;
                  case 0xc051 : if (virtgraphic) {
                                  virtgraphic = 0;
                                  virtsetmode();
                                }
                                return virtvideobyte;
                  case 0xc052 : if (virtmixed) {
                                  virtmixed = 0;
                                  virtsetmode();
                                }
                                return virtvideobyte;
                  case 0xc053 : if (!virtmixed) {
                                  virtmixed = 0x80;
                                  virtsetmode();
                                }
                                return virtvideobyte;
                  case 0xc054 : if (virtpage2) {
                                  virtpage2 = 0;
                                  if (!memstore80) {
                                    virtcachemode = (((virtcachemode & 0xfe) + 2) & 0xfe) | (virtcachemode & 0xff00);
                                    virtsetmode();
                                  }
                                }
                                return virtvideobyte;
                  case 0xc055 : if (!virtpage2) {
                                  virtpage2 = 0x80;
                                  if (!memstore80) {
                                    virtsetmode();
                                  }
                                }
                                return virtvideobyte;
                  case 0xc056 : if (virthgr) {
                                  virthgr = 0;
                                  virtsetmode();
                                }
                                return virtvideobyte;
                  case 0xc057 : if (!virthgr) {
                                  virthgr = 0x80;
                                  virtsetmode();
                                }
                                return virtvideobyte;
/* C058 49240 CLRAN0       OE G WR   If IOUDIS off: Annunciator 0 Off
              DISXY          C  WR   If IOUDIS on: Mask X0/Y0 Move Interrupts */
                  case 0xc058 : if (!virtiou) {
                                  memann0 = 0;
                                }
                                else {
                                  if (appletype & APPLEIIC) {
                                    return mousereadiou(addr);
                                  }
                                }
                                return virtvideobyte;
/* C059 49241 SETAN0       OE G WR   If IOUDIS off: Annunciator 0 On
              ENBXY          C  WR   If IOUDIS on: Allow X0/Y0 Move Interrupts */
                  case 0xc059 : if (!virtiou) {
                                  memann0 = 0x80;
                                }
                                else {
                                  if (appletype & APPLEIIC) {
                                    return mousereadiou(addr);
                                  }
                                }
                                return virtvideobyte;
/* C05A 49242 CLRAN1       OE G WR   If IOUDIS off: Annunciator 1 Off
              DISVBL         C  WR   If IOUDIS on: Disable VBL Interrupts */
                  case 0xc05a : if (!virtiou) {
                                  memann1 = 0;
                                }
                                else {
                                  if (appletype & APPLEIIC) {
                                    return mousereadiou(addr);
                                  }
                                }
                                return virtvideobyte;
/* C05B 49243 SETAN1       OE G WR   If IOUDIS off: Annunciator 1 On
              ENVBL          C  WR   If IOUDIS on: Enable VBL Interrupts */
                  case 0xc05b : if (!virtiou) {
                                  memann1 = 0x80;
                                }
                                else {
                                  if (appletype & APPLEIIC) {
                                    return mousereadiou(addr);
                                  }
                                }
                                return virtvideobyte;
/* C05C 49244 CLRAN2       OE G WR   If IOUDIS off: Annunciator 2 Off
              X0EDGE         C  WR   If IOUDIS on: Interrupt on X0 Rising */
                  case 0xc05c : if (!virtiou) {
                                  memann2 = 0;
                                }
                                else {
                                  if (appletype & APPLEIIC) {
                                    return mousereadiou(addr);
                                  }
                                }
                                return virtvideobyte;
/* C05D 49245 SETAN2       OE G WR   If IOUDIS off: Annunciator 2 On
              X0EDGE         C  WR   If IOUDIS on: Interrupt on X0 Falling */
                  case 0xc05d : if (!virtiou) {
                                  memann2 = 0x80;
                                }
                                else {
                                  if (appletype & APPLEIIC) {
                                    return mousereadiou(addr);
                                  }
                                }
                                return virtvideobyte;
/* C05E 49246 CLRAN3       OE G WR   If IOUDIS off: Annunciator 3 Off
              Y0EDGE         C  WR   If IOUDIS on: Interrupt on Y0 Rising
              DHIRESON      ECG WR   In 80-Column Mode: Double Width Graphics */
                  case 0xc05e : if (!virtdhres) {
                                  virtdhres = 0x80;         /* double hires on   */
                                  virtsetmode();
                                }
                                if (!virtiou) {
                                  memann3 = 0;              /* annunciator#3 off */
                                }
                                else {
                                  if (appletype & APPLEIIC) {
                                    return mousereadiou(addr);
                                  }
                                }
                                return virtvideobyte;
/*  C05F 49247 SETAN3       OE G WR   If IOUDIS off: Annunciator 3 On
               Y0EDGE         C  WR   If IOUDIS on: Interrupt on Y0 Falling
               DHIRESOFF     ECG WR   In 80-Column Mode: Single Width Graphics */
                  case 0xc05f : if (virtdhres) {
                                  virtdhres = 0;            /* double hires off */
                                  virtsetmode();
                                }
                                if (!virtiou) {
                                  memann3 = 0x80;           /* annunciator#3 on */
                                }
                                else {
                                  if (appletype & APPLEIIC) {
                                    return mousereadiou(addr);
                                  }
                                }
                                return virtvideobyte;
/* C060 49248              OE    R7  Read Cassette Input
                             C   R7  Status of 80/40 Column Switch
              BUTN3           G  R7  Switch Input 3 */
                  case 0xc060 : if (appletype & APPLEIIC) {
                                  if (memiic4080) {
                                    return virtvideobyte & 0x7f; /* 80 */
                                  }
                                  return virtvideobyte | 0x80; /* 40 */
                                }
                                else {
                                  return virtvideobyte & 0x7f;
                                }
                  case 0xc068 : return 0x00; /* Cassette input */
/* C061 49249 RDBTN0        EC   R7  Switch Input 0 / Open Apple */
                  case 0xc061 :
                  case 0xc069 : return memjoybutton0;           /* here: also possible on AppleII */
/* C062 49250 BUTN1         E    R7  Switch Input 1 / Solid Apple */
                  case 0xc062 :
                  case 0xc06a : return memjoybutton1;           /* here: also possible on AppleII */
/* C063 49251 RD63          E    R7  Switch Input 2 / Shift Key
                             C   R7  Bit 7 = Mouse Button Not Pressed */
/* Switched values, and added the ability to use this on the ][+.  It is
   possible to have the Shift Key Mod on any ][ - it was just *standard*
   on the Platinum //e only. -uso. */
                  case 0xc063 :
                  case 0xc06b : if (appletype & (APPLEII | APPLEIIE)) {
                                  if ((keyboardkeymap[0x80]) || (keyboardkeymap[0x81])) {
                                    return 0x00;
                                  }
                                  else {
                                    return 0x80;
                                  }
                                }
                                else {
                                  return (mousebutton & 0xff) ? 0x00 : 0x80;
                                }
/* C064 49252 PADDL0       OECG  R7  Analog Input 0 */
                  case 0xc064 :
                  case 0xc06c : if ((cpugetcycle() - memjoycycle - memjoyx) > 0x7fffffff) {
                                  return 0x80;
                                }
                                else {
                                  return 0;
                                }
/* C065 49253 PADDL1       OECG  R7  Analog Input 1 */
                  case 0xc065 :
                  case 0xc06d : if ((cpugetcycle() - memjoycycle - memjoyy) > 0x7fffffff) {
                                  return 0x80;
                                }
                                else {
                                  return 0;
                                }
/* C066 49254 PADDL2       OE G  R7  Analog Input 2
              RDMOUX1        C   R7  Mouse Horiz Position */
                  case 0xc066 :
                  case 0xc06e : return 0xa0; /* Pdl2 */
/* C067 49255 PADDL3       OE G  R7  Analog Input 3
              RDMOUY1        C   R7  Mouse Vert Position */
                  case 0xc067 :
                  case 0xc06f : return 0xa0; /* Pdl3 */

/* C070 49364 PTRIG         E    R   Analog Input Reset
                             C  WR   Analog Input Reset + Reset VBLINT Flag */
                  case 0xc070 : if (keyjoystick) {
                                  union REGS regs;

                                  regs.h.ah=0x84;       /* Get the current position of the game controller */
                                  regs.x.dx=0x0001;
                                  int86(0x15,&regs,&regs);
                                  memjoyx=(27*(regs.x.ax&0xFFFF));
                                  memjoyy=(27*(regs.x.bx&0xFFFF));
                                }
                                memjoycycle = cpugetcycle();    /* save cpu cycle */
                                return virtvideobyte;
                  case 0xc071 :
                  case 0xc072 :
                  case 0xc073 :
                  case 0xc074 :
                  case 0xc075 :
                  case 0xc076 :
                  case 0xc077 :
                  case 0xc078 :
                  case 0xc079 :
                  case 0xc07a :
                  case 0xc07b :
                  case 0xc07c :
                  case 0xc07d : return virtvideobyte;

/* C07E 49278 IOUDISON      EC  W    Disable IOU
              RDIOUDIS      EC   R7  Status of IOU Disabling */
                  case 0xc07e : if (appletype & (APPLEIIE | APPLEIIC)) {
                                  return ((virtiou) | (virtvideobyte & 0x7f));
                                }
                                else {
                                  return virtvideobyte;
                                }
/* C07F 49279 IOUDISOFF     EC  W    Enable IOU
              RDDHIRES      EC   R7  Status of Double HiRes */
                  case 0xc07f : if (appletype & (APPLEIIE | APPLEIIC)) {
                                  return ((virtdhres) | (virtvideobyte & 0x7f));
                                }
                                else {
                                  return virtvideobyte;
                                }

/* Language Card Handling */
                  case 0xc080 :
                  case 0xc084 : memlcbank2 = 0x80;      /* Bank 2 */
                                memlcramr  = 0x80;      /* Read Bank */
                                memlcramw  = 0;         /* Write Rom */
                                return (memkey&0x7f);
                  case 0xc081 :
                  case 0xc085 : memlcbank2 = 0x80;      /* Bank 2 */
                                memlcramr  = 0;         /* Read Rom */
                                memlcramw  = 0x80;      /* Write Bank */
                                return (memkey&0x7f);
                  case 0xc082 :
                  case 0xc086 : memlcbank2 = 0x80;      /* Bank 2 */
                                memlcramr  = 0;         /* Read Rom */
                                memlcramw  = 0;         /* Write Rom */
                                return (memkey&0x7f);
                  case 0xc083 :
                  case 0xc087 : memlcbank2 = 0x80;      /* Bank 2 */
                                memlcramr  = 0x80;      /* Read Bank */
                                memlcramw  = 0x80;      /* Write Bank */
                                return (memkey&0x7f);
                  case 0xc088 :
                  case 0xc08c : memlcbank2 = 0;         /* Bank 1 */
                                memlcramr  = 0x80;      /* Read Bank */
                                memlcramw  = 0;         /* Write Rom */
                                return (memkey&0x7f);
                  case 0xc089 :
                  case 0xc08d : memlcbank2 = 0;         /* Bank 1 */
                                memlcramr  = 0;         /* Read Rom */
                                memlcramw  = 0x80;      /* Write Bank */
                                return (memkey&0x7f);
                  case 0xc08a :
                  case 0xc08e : memlcbank2 = 0;         /* Bank 1 */
                                memlcramr  = 0;         /* Read Rom */
                                memlcramw  = 0;         /* Write Rom */
                                return (memkey&0x7f);
                  case 0xc08b :
                  case 0xc08f : memlcbank2 = 0;         /* Bank 1 */
                                memlcramr  = 0x80;      /* Read Bank */
                                memlcramw  = 0x80;      /* Write Bank */
                                return (memkey&0x7f);
                } /* switch (addr) */
              } /* if (addr <= 0xc08f) */

/* Slot #1 Softswitches: Parallel Port */
        if (addr <= 0xc09f) {
          if (appletype & (APPLEII | APPLEIIE)) {
            return slots[1].slotget(slots[1].slotdata, addr);
          }
          return virtvideobyte;
        }

/* Slot #2 Softswitches: Mouse */
        if (addr <= 0xc0af) {
          if (appletype & (APPLEII | APPLEIIE)) {       /* so far, //c mouse is not emulated */
            return slots[2].slotget(slots[2].slotdata, addr);
          }
          else {
            return virtvideobyte;
          }
        }

/* Slot #3 Softswitches: 80col */
        if (addr <= 0xc0bf) {
          if (appletype & (APPLEII | APPLEIIE)) {       /* so far, //c mouse is not emulated */
            return slots[3].slotget(slots[3].slotdata, addr);
          }
          else {
            return virtvideobyte;
          }
        }

/* Slot #4 Softswitches: Z80 */
        if (addr <= 0xc0cf) {
          if (appletype & (APPLEII | APPLEIIE)) {       /* put in Z80 card here */
            return slots[4].slotget(slots[4].slotdata, addr);
          }
          else {
            return virtvideobyte;
          }
        }

/* Slot #5 Softswitches */
        if (addr <= 0xc0df) {
          if (appletype & (APPLEII | APPLEIIE)) {
            return slots[5].slotget(slots[5].slotdata, addr);
          }
          else {
            return virtvideobyte;
          }
        }

/* Slot #6 Softswitches */
        if (addr >= 0xc0e0 && addr <= 0xc0ef) {
          return slots[6].slotget(slots[6].slotdata, addr);
        }

/* Slot #7 Softswitches */
        if (addr >= 0xc0f0 && addr <= 0xc0ff) {
          if (appletype & (APPLEII | APPLEIIE)) {
            return virtvideobyte; /* dqhdv? virtvideobyte : ReadMassStorIO (addr); */
          }
          else {
            return virtvideobyte;
          }
        }

/* The remaining addresses between 0xc000 and 0xc0ff are simply ignored. */
        return virtvideobyte;
      } /* if (addr <= 0xc0ff */
      else {
/* $c100 - $cfff */
        if ((memintcxrom) || (memrombank)) {
          return memram[(addr-0xc000)+MEMINTCXROM];
        }
        else {

          if (addr <= 0xc1ff) {                                                 /* 1 */
            if (appletype & (APPLEII | APPLEIIE)) {
              return slots[1].slotromget(slots[1].slotdata, addr);
            }
            else {
              return memram[(addr-0xc000)+MEMSLOTROM];                          /* //c rom */
            }
          }

          if (addr <= 0xc2ff) {                                                 /* 2 */
            if (appletype & (APPLEII | APPLEIIE)) {
              return slots[2].slotromget(slots[2].slotdata, addr);
            }
            else {
              return memram[(addr-0xc000)+MEMSLOTROM];                          /* //c rom */
            }
          }

          if (addr <= 0xc3ff) {                                                 /* 3 */
            if (appletype & APPLEIIC) {
              return memram[(addr-0xc000)+MEMSLOTROM];                          /* //c rom */
            }
            else {
              if ((appletype & APPLEIIE) && (!memslotc3)) {
                return memram[(addr-0xc000)+MEMINTCXROM];               /* copy of INTCXROM */
              }
              else {
                return slots[3].slotromget(slots[3].slotdata, addr);    /* slot */
              }
            }
          }

          if (addr <= 0xc4ff) {                                                 /* 4 */
            if (appletype & (APPLEII | APPLEIIE)) {
              return slots[4].slotromget(slots[4].slotdata, addr);
            }
            else {
              return memram[(addr-0xc000)+MEMSLOTROM];                          /* //c rom */
            }
          }

          if (addr <= 0xc5ff) {                                                 /* 5 */
            if (appletype & (APPLEII | APPLEIIE)) {
              return slots[5].slotromget(slots[5].slotdata, addr);
            }
            else {
              return memram[(addr-0xc000)+MEMSLOTROM];                          /* //c rom */
            }
          }

          if (addr <= 0xc6ff) {                                                 /* 6 */
            if (appletype & (APPLEII | APPLEIIE)) {
              return slots[6].slotromget(slots[6].slotdata, addr);
            }
            else {
              return memram[(addr-0xc000)+MEMSLOTROM];                          /* //c rom */
            }
          }

          if (addr <= 0xc7ff) {                                                 /* 7 */
            if (appletype & (APPLEII | APPLEIIE)) {
              return slots[7].slotromget(slots[7].slotdata, addr);
            }
            else {
              return memram[(addr-0xc000)+MEMSLOTROM];                          /* //c rom */
            }
          }

          if (appletype & APPLEII) {
            return 0xff;
          }
          else {
            if (appletype & APPLEIIE) {
              return memram[(addr-0xc000)+MEMINTCXROM];
            }
            else {
              return memram[(addr-0xc000)+MEMSLOTROM];
            }
          }
        }
      } /* else if (addr <= 0xc0ff) */
    } /* else if (addr >= 0xd000) */
  } /* else if (addr < 0xc000) */
} /* memoryread() */


/*-------------------------------------*/


      void memorywrite(unsigned int addr, unsigned int val) {

        if (addr < 0xc000) {
          if (addr < 0x2000) {
            if (addr < 0x400) {
              if (addr < 0x200) {
/* $0000 - $01ff */
                if (memlcaux)   { memauxbankptr[addr]   = val; return; }        /* aux 64 */
                else            { memram[addr]          = val; return; }        /* main 64 */
              }
              else {
/* $0200 - $03ff */
                if (memauxwrite){ memauxbankptr[addr]   = val; return; }        /* aux 64 */
                else            { memram[addr]          = val; return; }        /* main 64 */
              }
            }
            else {
              if (addr < 0x800) {
/* $0400 - $07ff */
                if (memstore80) {
                  if (virtpage2) {
                    if (memram[addr+MEMAUXRAM] != val) {
                      memram[addr+MEMAUXRAM] = val;                             /* aux 64 */
                      virtwrite0400aux(addr);
                    }
                  }
                  else {
                    if (memram[addr] != val) {
                      memram[addr] = val;                                       /* main 64 */
                      virtwrite0400(addr);
                    }
                    return;
                  }
                } /* if (memstore80) */
                else {
                  if (memauxwrite) {
                    if (!memauxbank) {
                      if (memram[addr+MEMAUXRAM] != val) {
                        memram[addr+MEMAUXRAM] = val;                           /* aux 64 */
                        virtwrite0400aux(addr);
                      }
                    }
                    else {
                      memauxbankptr[addr] = val;
                    }
                    return;
                  }
                  else {
                    if (memram[addr] != val) {
                      memram[addr] = val;                                       /* main 64 */
                      virtwrite0400(addr);
                    }
                    return;
                  }
                } /* else if (memstore80) */
              } /* if (addr < 0x800 */
              else {
                if (addr < 0xc00) {
/* $0800 - $0bff */
                  if (memauxwrite) {
                    if (!memauxbank) {
                      if (memram[addr+MEMAUXRAM] != val) {
                        memram[addr+MEMAUXRAM] = val;                           /* aux 64 */
                        virtwrite0800aux(addr);
                      }
                    }
                    else {
                      memauxbankptr[addr] = val;
                    }
                    return;
                  }
                  else {
                    if (memram[addr] != val) {
                      memram[addr] = val;                                       /* main 64 */
                      virtwrite0800(addr);
                    }
                    return;
                  }
                }
                else {
/* $0c00 - $1fff */
                  if (memauxwrite)      { memauxbankptr[addr]   = val; return; }        /* aux 64 */
                  else                  { memram[addr]          = val; return; }        /* main 64 */
                }
              }
            }
          } /* if (addr < 0x2000) */
          else {
            if (addr < 0x4000) {
/* $2000 - $3fff */
              if ((memstore80) && (virthgr)) {
                if (virtpage2) {
                  if (memram[addr+MEMAUXRAM] != val) {
                    memram[addr+MEMAUXRAM] = val;                               /* aux 64 */
                    virtwrite2000aux(addr);
                  }
                  return;
                }
                else {
                  if (memram[addr] != val) {
                    memram[addr] = val;                                         /* main 64 */
                    virtwrite2000(addr);
                  }
                  return;
                }
              }
              else {
                if (memauxwrite) {
                  if (!memauxbank) {
                    if (memram[addr+MEMAUXRAM] != val) {
                      memram[addr+65536] = val;                                 /* aux 64 */
                      virtwrite2000aux(addr);
                    }
                  }
                  else {
                    memauxbankptr[addr] = val;
                  }
                  return;
                }
                else {
                  if (memram[addr] != val) {
                    memram[addr] = val;                                         /* main 64 */
                    virtwrite2000(addr);
                  }
                  return;
                }
              }
            }
            else {
              if (addr < 0x6000) {
/* $4000 - $5fff */
                if (memauxwrite) {
                  if (!memauxbank) {
                    if (memram[addr+MEMAUXRAM] != val) {
                      memram[addr+MEMAUXRAM] = val;                             /* aux 64 */
                      virtwrite4000aux(addr);
                    }
                  }
                  else {
                    memauxbankptr[addr] = val;
                  }
                  return;
                }
                else {
                  if (memram[addr] != val) {
                    memram[addr] = val;                                         /* main 64 */
                    virtwrite4000(addr);
                  }
                  return;
                }
              }
              else {
/* $6000 - $bfff */
                if (memauxwrite)        { memauxbankptr[addr]   = val; return; }/* aux 64 */
                else                    { memram[addr]          = val; return; }/* main 64 */
              }
            }
          }
        } /* if (addr < 0xc000) */
        else {
          if (addr >= 0xd000) {
            if ((!memlcramw) || (memnolc && (appletype & APPLEII)) ) {
              return;         /* ROM is the same for whatever memlcaux is set to */
            }
            else {
              if (memlcaux) {
                if (addr < 0xe000) {
                  if (!memlcbank2) {
                     memauxbankptr[(addr-0x1000)] = val;
                    return;
                  }
                }
                memauxbankptr[addr] = val;
                return;
              }
              else {
                if (addr < 0xe000) {
                  if (!memlcbank2) {
                    memram[addr-0x1000] = val;
                    return;
                  }
                }
                memram[addr] = val;
                return;
              }
            } /* else if (memlcramw) */
          } /* if (addr => 0xd000 */
          else {

 /* test for softswitch area */
            if (addr <= 0xc0ff) {

              if (inidebugflag) return;

                if (addr <= 0xc00f) {
                  if (appletype & (APPLEIIE | APPLEIIC | LASER128)) {   /* not on Apple][+ */
                    switch (addr) {
                      case 0xc000 : if (memstore80) {
                                      memstore80 = 0;
                                      virtsetmode();
                                    }
                                    return;
                      case 0xc001 : if (!memstore80) {
                                      memstore80 = 0x80;
                                      virtsetmode();
                                    }
                                    return;
                      case 0xc002 : memauxread          = 0;    return;
                      case 0xc003 : memauxread          = 0x80; return;
                      case 0xc004 : memauxwrite         = 0;    return;
                      case 0xc005 : memauxwrite         = 0x80; return;
                      case 0xc006 : if (appletype & (APPLEIIE | LASER128)) {
                                      memintcxrom       = 0;                    /* SLOTCXROM Off */
                                    }
                                    return;
                      case 0xc007 : if (appletype & (APPLEIIE | LASER128)) {
                                      memintcxrom       = 0x80;                 /* SLOTCXROM On */
                                    }
                                    return;
                      case 0xc008 : memlcaux            = 0;    return;
                      case 0xc009 : memlcaux            = 0x80; return;
                      case 0xc00a : if (appletype & APPLEIIE) {
                                      memslotc3         = 0;                    /* SLOTC3ROM Off */
                                    }
                                    return;
                      case 0xc00b : if (appletype & APPLEIIE) {
                                      memslotc3         = 0x80;                 /* SLOTC3ROM On */
                                    }
                                    return;
                      case 0xc00c : if (virt80col) {
                                      virt80col = 0;
                                      virtsetmode();
                                    }
                                    return;
                      case 0xc00d : if (!virt80col) {
                                      virt80col = 0x80;
                                      virtsetmode();
                                    }
                                    return;
                      case 0xc00e : if (virtaltchar) {                  /* only update if there was a change */
                                      virtaltchar = 0;
                                      virtfontptr = virtfontflashptr;
                                      virtcacheinit();
                                    }
                                    return;
                      case 0xc00f : if (!virtaltchar) {                 /* only update if there was a change */
                                      virtaltchar = 0x80;
                                      virtfontptr = virtfontalt;
                                      virtcacheinit();
                                    }
                                    return;
                    } /* switch (addr) */
                  } /* if (appletype & (APPLEIIE | APPLEIIC | LASER128)) */
                  else {        /* APPLEII */
                    return;
                  }
                } /* if (addr <= 0xc00f */

                if (addr <= 0xc01f) {
                  memkey = memkey & 0x7f;
                  return;
                }

                if (addr <= 0xc02f) {
                  if (appletype & APPLEIIC) {
                    if (addr == 0xc028) {
                      memrombank = !memrombank;
                    }
                  }
                  return;                               /* Cassette output toggle */
                }

                if (addr <= 0xc03f)     { soundclick(); /*soundclick();*/ return; } /* Speaker */

                if (addr <= 0xc04f)     return;         /* Utility output strobe - nani yo? */

                if (addr <= 0xc08f) {
                  switch (addr) {
                    case 0xc050 :
                      if (!virtgraphic) {
                        virtgraphic = 0x80;
                        virtsetmode();
                      }
                      return;
                    case 0xc051 :
                      if (virtgraphic) {
                        virtgraphic = 0;
                        virtsetmode();
                      }
                      return;
                    case 0xc052 :
                      if (virtmixed) {
                        virtmixed = 0;
                        virtsetmode();
                      }
                     return;
                    case 0xc053 :
                      if (!virtmixed) {
                        virtmixed = 0x80;
                        virtsetmode();
                      }
                      return;
                    case 0xc054 :
                      if (virtpage2) {
                        virtpage2 = 0;
                        if (!memstore80) {
                          virtcachemode = (((virtcachemode & 0xfe) + 2) & 0xfe) | (virtcachemode & 0xff00);
                          virtsetmode();
                        }
                      }
                      return;
                    case 0xc055 :
                      if (!virtpage2) {
                        virtpage2 = 0x80;
                        if (!memstore80) {
                          virtsetmode();
                        }
                      }
                      return;
                    case 0xc056 :
                      if (virthgr) {
                        virthgr = 0;
                        virtsetmode();
                      }
                      return;
                    case 0xc057 :
                      if (!virthgr) {
                        virthgr = 0x80;
                        virtsetmode();
                      }
                      return;

                    case 0xc058 : return; /* Out0-off - nani yo? */
                    case 0xc059 : return; /* Out0-on - nani yo? */
                    case 0xc05a : return; /* Out1-off - nani yo? */
                    case 0xc05b : return; /* Out1-on - nani yo? */
                    case 0xc05c : return; /* Out2-off - nani yo? */
                    case 0xc05d : return; /* Out2-on - nani yo? */
/*
   C05E 49246 CLRAN3       OE G WR   If IOUDIS off: Annunciator 3 Off
              Y0EDGE         C  WR   If IOUDIS on: Interrupt on Y0 Rising
              DHIRESON      ECG WR   In 80-Column Mode: Double Width Graphics
*/
            case 0xc05e : {
                            if (!virtiou) {
                              memann3 = 0;              /* annunciator#3 off */
                            }
                            if (!virtdhres) {
                              virtdhres = 0x80; /* double hires on   */
                              virtsetmode();
                            }
                            return;
                          }
/*
    C05F 49247 SETAN3       OE G WR   If IOUDIS off: Annunciator 3 On
               Y0EDGE         C  WR   If IOUDIS on: Interrupt on Y0 Falling
               DHIRESOFF     ECG WR   In 80-Column Mode: Single Width Graphics
*/
            case 0xc05f : {
                            if (!virtiou) {
                              memann3 = 0x80;           /* annunciator#3 on */
                            }
                            if (virtdhres) {
                              virtdhres = 0;            /* double hires off */
                              virtsetmode();
                            }
                            return;
                          }
            case 0xc060 :
            case 0xc068 : return; /* Cassette input */
            case 0xc061 :
            case 0xc069 : return;
            case 0xc062 :
            case 0xc06a : return;
/* C063 49251 RD63          E    R7  Switch Input 2 / Shift Key
                             C   R7  Bit 7 = Mouse Button Not Pressed */
            case 0xc063 :
            case 0xc06b : return;
            case 0xc064 :
            case 0xc06c : return; /* Pdl0 */
            case 0xc065 :
            case 0xc06d : return; /* Pdl1 */
            case 0xc066 :
            case 0xc06e : return; /* Pdl2 */
            case 0xc067 :
            case 0xc06f : return; /* Pdl3 */

            case 0xc070 :
              if (keyjoystick) {
                union REGS regs;
                regs.h.ah=0x84;         /* Get the current position of the game controller */
                regs.x.dx=0x0001;
                int86(0x15,&regs,&regs);
                memjoyx=(27*(regs.x.ax&0xFFFF));
                memjoyy=(27*(regs.x.bx&0xFFFF));
              }
              memjoycycle = cpugetcycle();      /* save cpu cycle */
              return;
            case 0xc071 : return;
            case 0xc072 : return;
/* C073 49367 BANKSEL       ECG W    Memory Bank Select for > 128K */
            case 0xc073:
              if (appletype & (APPLEIIE | APPLEIIC)) {
                if (val < MEMMAXRAM) {  // nani? what is one to do upon hitting an invalid page?
                  memauxbank = val;
                  if (memramptr[val] != NULL) {
                    memauxbankptr = memramptr[val];
                  }
                }
              }
              return;
            case 0xc074 :
            case 0xc075 :
            case 0xc076 :
            case 0xc077 :
              return;
/* C078 49372                C  W    Disable IOU Access */
            case 0xc078 :
              if (appletype & APPLEIIC) {
                virtiou = 0;
              }
              return;
/* C079 49373                C  W    Enable IOU Access */
            case 0xc079 :
              if (appletype & APPLEIIC) {
                virtiou = 0x80;
              }
            case 0xc07a :
            case 0xc07b :
            case 0xc07c :
            case 0xc07d :
              return;
/* C07E 49278 IOUDISON      EC  W    Disable IOU
              RDIOUDIS      EC   R7  Status of IOU Disabling */
            case 0xc07e : if (appletype & (APPLEIIE | APPLEIIC)) {
                            virtiou = 0;
                          }
                          return;
/* C07F 49279 IOUDISOFF     EC  W    Enable IOU
              RDDHIRES      EC   R7  Status of Double HiRes */
            case 0xc07f : if (appletype & (APPLEIIE | APPLEIIC)) {
                            virtiou = 0x80;
                          }
                          return;

/* Language Card Handling */
            case 0xc080 :
            case 0xc084 :
              memlcbank2 = 0x80;  /* Bank 2 */
              memlcramr  = 0x80;  /* Read Bank */
              memlcramw  = 0;     /* Write Rom */
              return;
            case 0xc081 :
            case 0xc085 :
            memlcbank2 = 0x80;  /* Bank 2 */
              memlcramr  = 0;     /* Read Rom */
              memlcramw  = 0x80;  /* Write Bank */
              return;
            case 0xc082 :
            case 0xc086 :
              memlcbank2 = 0x80;  /* Bank 2 */
              memlcramr  = 0;     /* Read Rom */
              memlcramw  = 0;     /* Write Rom */
              return;
            case 0xc083 :
            case 0xc087 :
              memlcbank2 = 0x80;  /* Bank 2 */
              memlcramr  = 0x80;  /* Read Bank */
              memlcramw  = 0x80;  /* Write Bank */
              return;
            case 0xc088 :
            case 0xc08c :
              memlcbank2 = 0;     /* Bank 1 */
              memlcramr  = 0x80;  /* Read Bank */
              memlcramw  = 0;     /* Write Rom */
              return;
            case 0xc089 :
            case 0xc08d :
              memlcbank2 = 0;     /* Bank 1 */
              memlcramr  = 0;     /* Read Rom */
              memlcramw  = 0x80;  /* Write Bank */
              return;
            case 0xc08a :
            case 0xc08e :
              memlcbank2 = 0;     /* Bank 1 */
              memlcramr  = 0;     /* Read Rom */
              memlcramw  = 0;     /* Write Rom */
              return;
            case 0xc08b :
            case 0xc08f :
              memlcbank2 = 0;     /* Bank 1 */
              memlcramr  = 0x80;  /* Read Bank */
              memlcramw  = 0x80;  /* Write Bank */
              return;
          } /* switch */
        }

/* Slot #1 Softswitches: Parallel Port */
        if (addr <= 0xc09f) {
          if (appletype & (APPLEII | APPLEIIE)) {       /* not on Apple//c */
            slots[1].slotset(slots[1].slotdata, addr, val);
          }
          return;
        }

/* Slot #2 Softswitches: Mouse */
        if (addr <= 0xc0af) {
          if (appletype & (APPLEII | APPLEIIE)) {       /* so far, //c mouse is not emulated */
            slots[2].slotset(slots[2].slotdata, addr, val);
          }
          return;
        }

/* Slot #3 Softswitches: 80col */
        if (addr <= 0xc0bf) {
          if (appletype & (APPLEII | APPLEIIE)) {
            slots[3].slotset(slots[3].slotdata, addr, val);
          }
          return;
        }

/* Slot #4 Softswitches: Z80 */
        if (addr <= 0xc0cf) {
          if (appletype & (APPLEII | APPLEIIE)) {
            slots[4].slotset(slots[4].slotdata, addr, val);
          }
          return;
        }

/* Slot #5 Softswitches */
        if (addr <= 0xc0df) {
          if (appletype & (APPLEII | APPLEIIE)) {       /* not on Apple//c */
            slots[5].slotset(slots[5].slotdata, addr, val);
          }
          return;
        }

/* Slot #6 Softswitches */
        if (addr <= 0xc0ef) {
          slots[6].slotset(slots[6].slotdata, addr, val);
          return;
        }

/* Slot #7 Softswitches */
        if (addr <= 0xc0ff) {
          return;
        }
/*      if (addr >= 0xc0f0 && addr <= 0xc0ff && (!dqhdv)) {
          WriteMassStorIO (addr, val);  return;
        }
*/

/* The remaining addresses between 0xc000 and 0xc0ff are simply ignored. */
          return;
        } /* if (addr <= 0xc0ff) */

            if (memintcxrom) { return; }

            if ((addr >= 0xc100) && (addr <= 0xc1ff)) {
              if (appletype & (APPLEII | APPLEIIE)) {
                slots[1].slotromset(slots[1].slotdata, addr, val);
              }
              return;
            }

/* Slot #2 : Mouse */
            if ((addr >= 0xc200) && (addr <= 0xc2ff)) {
              if (appletype & (APPLEII | APPLEIIE)) {
                slots[2].slotromset(slots[2].slotdata, addr, val);
              }
              return;
            }

/* Slot #3 */
            if ((addr >= 0xc300) && (addr <= 0xc3ff)) {
              if (appletype & (APPLEII | APPLEIIE)) {
                slots[3].slotromset(slots[3].slotdata, addr, val);
              }
              return;
            }

/* Slot #4 : Z80 */
            if ((addr >= 0xc400) && (addr <= 0xc4ff)) {
              if (appletype & (APPLEII | APPLEIIE)) {
                slots[4].slotromset(slots[4].slotdata, addr, val);
              }
              return;
            }

/* Slot #5 */
            if ((addr >= 0xc500) && (addr <= 0xc5ff)) {
              if (appletype & (APPLEII | APPLEIIE)) {
                slots[5].slotromset(slots[5].slotdata, addr, val);
              }
              return;
            }

/* Slot #6 */
            if ((addr >= 0xc600) && (addr <= 0xc6ff)) {
              if (appletype & (APPLEII | APPLEIIE)) {
                slots[6].slotromset(slots[6].slotdata, addr, val);
              }
              return;
            }

/* Slot #7 */
            if ((addr >= 0xc700) && (addr <= 0xc7ff)) {
              if (appletype & (APPLEII | APPLEIIE)) {
                slots[7].slotromset(slots[7].slotdata, addr, val);
              }
              return;
            }

            return;
          } // else (if addr >= 0xd000
        } // else (if (addr < 0xc000
      } // memorywrite


/*-------------------------------------*/


      void memorymenu(void) {
        unsigned int  menukeyboard;
        unsigned int  menuwindow;
        unsigned char key;
        unsigned int  update;
        unsigned char filename[260];

        if (!windowaddio( -1, -1, WINDOWXSIZE, WINDOWYSIZE, -1, 1,
"!\
EMemory Options;\
GSpeicher Optionen;\
", 0, &menukeyboard, &menuwindow)) {
          messageflag = 0;
          update = 1;
          do {
            if (update) {
              channelout(menuwindow, 12);               /* clear window */
              stringwritemessage(menuwindow,
"!\
E\r[ESC] - Quit\r\r\
Current ROM file is: ;\
G\r[ESC] - Verlasse Men\r\r\
Das aktuelle ROM ist: ;\
");
              stringwrite(menuwindow, memrompath);
              channelout(menuwindow, 13);

              stringwrite(menuwindow, "\r[1] - Apple ][ - Monitor              ");
              channelout(menuwindow,  0!=strstr(memrompath,"apple2m.rom")  ? '*': ' ');
              stringwrite(menuwindow, "\r[2] - Apple ][ - Autostart            ");
              channelout(menuwindow,  0!=strstr(memrompath,"apple2o.rom")  ? '*': ' ');
              stringwrite(menuwindow, "\r[3] - Apple ][ Plus                   ");
              channelout(menuwindow,  0!=strstr(memrompath,"apple.rom")    ? '*': ' ');
              stringwrite(menuwindow, "\r[4] - Apple //e (Original)            ");
              channelout(menuwindow,  0!=strstr(memrompath,"apple2eo.rom") ? '*': ' ');
              stringwrite(menuwindow, "\r[5] - Apple //e (Enhanced)            ");
              channelout(menuwindow,  0!=strstr(memrompath,"apple2e.rom") ? '*': ' ');
              stringwrite(menuwindow, "\r[6] - Apple //c (16 KB experimental!) ");
              channelout(menuwindow,  0!=strstr(memrompath,"apple2co.rom") ? '*': ' ');
              stringwrite(menuwindow, "\r[7] - Apple //c (32 KB experimental!) ");
              channelout(menuwindow,  0!=strstr(memrompath,"apple2c.rom")  ? '*': ' ');
              stringwritemessage(menuwindow,
"!\
E\r[F] - Select ROM file manually;\
G\r[F] - ROM file manuell auswhlen;\
");
              stringwrite(menuwindow, "\r\r[M] - Maximum ][/][+ RAM ");
              if (memnolc) { stringwrite(menuwindow, "48k"); }
              else         { stringwrite(menuwindow, "64k"); }

              stringwritemessage(menuwindow,
"!\
E\r(RAM in //e + //c mode is always ;\
G\r(RAM Speichergre im //e + //c Modus ist immer ;\
");
              int32write(menuwindow, (memnumbbanks+1) << 6);
              stringwrite(menuwindow, "KB)\r");

              stringwrite(menuwindow, "\r[T] - //c text 40/80 column switch ");
              stringwrite(menuwindow, memiic4080?"(80)\r":"(40)\r");

              stringwritemessage(menuwindow,
"!\
E\
\r[L] - Quickload emulator state\
\r[S] - Quicksave emulator state\
\r\r[B] - Bload/Brun binary file\
\r\r[R] - Reset computer\
\r[C] - Clear memory and reset computer\
\r[P] - Clear pattern = \
;\
G\
\r[L] - Lade Emulatorzustand\
\r[S] - Speichere Emulatorzustand\
\r\r[B] - Bload/Brun Binrdatei\
\r\r[R] - Reset Computer\
\r[C] - Lsche Speicher und reset Computer\
\r[P] - Lschmuster = \
;\
");
              switch (memorygetclearmode()) {
                case 1 :
                  stringwrite(menuwindow, "00 00 00...");
                  break;
                case 2 :
                  stringwrite(menuwindow, "00 FF 00 FF...");
                  break;
                case 3 :
                  stringwrite(menuwindow, "00 FF ... FF 00...");
                  break;
                case 4 :
                  stringwrite(menuwindow, "EA EA EA...");
                  break;
                case 5 :
                  stringwrite(menuwindow, "00 01 02...");
                  break;
                case 6 :
                  stringwrite(menuwindow, "$DEADBEEF (6502 Little Endian)");
                  break;
                case 7 :
                  stringwrite(menuwindow, "$DEADBEEF (High Endian)");
                  break;
              } // switch
              update = 0;
              screenupdate = 1;
            }
            do {
              taskswitch();
              if (windowgetclose(menuwindow)) {
                key = 27;
              }
              else {
                key = (unsigned char)channelin(menukeyboard);
              }
            }
            while ((key == 0) && (!exitprogram));
            switch (key) {
              case '1' :
                channelout(menuwindow, 12);
                strcpy(filename, inirompath);
                strcat(filename, "\\apple2m.rom");
                memoryloadrom(menukeyboard, menuwindow, filename);
                update = 1;
                break;
              case '2' :
                channelout(menuwindow, 12);
                strcpy(filename, inirompath);
                strcat(filename, "\\apple2o.rom");
                memoryloadrom(menukeyboard, menuwindow, filename);
                update = 1;
                break;
              case '3' :
                channelout(menuwindow, 12);
                strcpy(filename, inirompath);
                strcat(filename, "\\apple2.rom");
                memoryloadrom(menukeyboard, menuwindow, filename);
                update = 1;
                break;
              case '4' :
                channelout(menuwindow, 12);
                strcpy(filename, inirompath);
                strcat(filename, "\\apple2eo.rom");
                memoryloadrom(menukeyboard, menuwindow, filename);
                update = 1;
                break;
              case '5' :
                channelout(menuwindow, 12);
                strcpy(filename, inirompath);
                strcat(filename, "\\apple2e.rom");
                memoryloadrom(menukeyboard, menuwindow, filename);
                update = 1;
                break;
              case '6' :
                channelout(menuwindow, 12);
                strcpy(filename, inirompath);
                strcat(filename, "\\apple2co.rom");
                memoryloadrom(menukeyboard, menuwindow, filename);
                update = 1;
                break;
              case '7' :
                channelout(menuwindow, 12);
                strcpy(filename, inirompath);
                strcat(filename, "\\apple2c.rom");
                memoryloadrom(menukeyboard, menuwindow, filename);
                update = 1;
                break;
              case 'f' :
              case 'F' :
                if (fileselectmenu(
"!\
ELoad Apple][ Rom;\
GLade Apple][ Rom;\
",
                                   "Apple][ ROM (*.ROM)",
                                   filename, inirompath, ".ROM", 0)) {
                  channelout(menuwindow, 12);
                  memoryloadrom(menukeyboard, menuwindow, filename);
                }
                update = 1;
                break;
              case 't' :                        /* 40/80 -uso. */
              case 'T' :
                memiic4080 = !memiic4080;
                update=1;
                break;

              case 'l' :
              case 'L' :
                if (fileselectmenu(
"!\
ELoad Emulator State;\
GLade Emulatorzustand;\
",
"!\
EEmulator State (*.SAV);\
GEmulatorzustand (*.SAV);\
",
                                   filename, inistatepath, ".SAV", 0)) {
                  applerestore(filename);
                }
                update = 1;
                break;
              case 's' :
              case 'S' :
                strcpy(filename, "");
                if (fileselectmenu(
"!\
ESave Emulator State;\
GSpeichere Emulatorzustand;\
",
"!\
EEmulator State (*.SAV);\
GEmulatorzustand (*.SAV);\
",
                                   filename, inistatepath, ".SAV", 1)) {
                  applestore(filename);
                }
                update = 1;
                break;

              case 'b' :
              case 'B' :
                strcpy(filename, "");
                if (fileselectmenu(
"!\
EBload/Brun Binary File;\
GBload/Brun Binrdatei;\
",
"!\
EBinary file (*.PG2, *.APL);\
GBinrdatei (*.PG2, *.APL);\
",
                                   filename, drivepath, "._prg_", 0)) {
                  memorybloadfile(filename);
                }
                update = 1;
                break;

              case 'm' :
              case 'M' :
                memnolc = !memnolc;
                update = 1;
                break;
              case 'p' :
              case 'P' :
                update = memorygetclearmode();
                update++;
                if (update > 7) { update = 1; }
                memorysetclearmode(update);
                update = 1;
                break;
              case 'r' :
              case 'R' :
                memoryresetmenu();
                break;
              case 'c' :
              case 'C' :
                memoryclearresetmenu();
                break;
            } // switch (key)
          }
          while ((key != 27) && (key != 32) && (key != 13) && (!exitprogram));
          channelclose(menukeyboard);
          channelclose(menuwindow);
          messageflag = 1;
        }

      } // memorymenu


// --> #ifndef DEF_INC_MEMORY_C
#endif
