/*
 * EMU][ Apple ][-class emulator
 * Copyright (C) 2002, 2003 by the EMU][ Project/Dapple ][ Team
 *
 * Component:  Z80: Z80 emulator
 *
 * This source file is not intended for release versions of EMU][ at this
 * time, because the lack of source to ASMLIB.O at this time prevents the
 * compliance with the license for YAZE (a small amount of code was taken
 * from YAZE and if it is used, there can be no exceptions to the GPL).
 * It is provided for debug builds.  We have intent to include this in the
 * future when full compliance with GPL is possible. -uso.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, version 2.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * 20040428  New headers generated for v0.4 release
 *           Z80 core development
 *
 */


/*----------------------------------------


        z80.c


        see include file z80.h for definitions


----------------------------------------*/


#ifndef DEF_INC_Z80_C
#define DEF_INC_Z80_C


// **** include general definition files

//#include <stdio.h>            // included by z80.h
//#include <string.h>           // included by z80.h
#include <unistd.h>             // used for chdir()
//#include "..\libs\general.h"  // included by z80.h
//#include "..\libs\asmlib.h"   // included by z80.h
//#include "dapple.h"           // included by z80.h


// **** include definition file of this source

#include "z80.h"


// **** include further Emu][ specific files

#include "cpu65c02.h"
#include "gui.h"
#include "memory.h"
#include "mouse.h"
#include "video.h"


// cpu65c02.c
void hex8tostringf(unsigned int value, unsigned char *stringpointer);




// **** local constants

#define CFLAG 0x01
#define NFLAG 0x02
#define PFLAG 0x04
#define FLAG3 0x08
#define HFLAG 0x10
#define FLAG5 0x20
#define ZFLAG 0x40
#define SFLAG 0x80


// **** local variables

static unsigned char    rega;
static unsigned char    regf;
static unsigned char    regb;
static unsigned char    regc;
static unsigned char    regd;
static unsigned char    rege;
static unsigned char    regh;
static unsigned char    regl;
static unsigned char    *regaddr[8] = { &regb, &regc, &regd, &rege, &regh, &regl, NULL, &rega };

static unsigned char    rega2;
static unsigned char    regf2;
static unsigned char    regb2;
static unsigned char    regc2;
static unsigned char    regd2;
static unsigned char    rege2;
static unsigned char    regh2;
static unsigned char    regl2;
static unsigned short   regix;
static unsigned short   regiy;
static unsigned char    regi;
static unsigned char    regr;
static unsigned short   regpc;
static unsigned short   regsp;
static unsigned char    regmux;

static unsigned char    regiff1;
static unsigned char    regiff2;

static unsigned int     z80cycle;               // Number of cycles
static unsigned int     z80lastcycle;
static unsigned int     z80linecycles;
static unsigned int     z80delay;

static unsigned int     z80stateflags;
static unsigned char    z80traceflag;
static unsigned int     z80breakpoint;

static unsigned char    z80rom[256];
static unsigned char    z80present;
unsigned char           z80active;

static unsigned char    flagssz [256];
static unsigned char    flagsszp[256];



/*----------------------------------------

 error  z80init (void)

        basic Z80 initialization

----------------------------------------*/

      unsigned char *z80init(void) {
        register unsigned int i;
        register unsigned int bitcount;

        rega    = 0;
        regf    = 0;
        regb    = 0;
        regc    = 0;
        regd    = 0;
        rege    = 0;
        regh    = 0;
        regl    = 0;
        rega2   = 0;
        regf2   = 0;
        regb2   = 0;
        regc2   = 0;
        regd2   = 0;
        rege2   = 0;
        regh2   = 0;
        regl2   = 0;
        regix   = 0x0000;
        regiy   = 0x0000;
        regpc   = 0x0000;
        regsp   = 0x0000;
        regi    = 0;
        regr    = 0;
        regmux  = 0;

        regiff1 = 0;
        regiff2 = 0;

        z80stateflags = CPU_STATEHALT;
        z80traceflag = 0;
        z80breakpoint = 0xffffffff;

        memset(z80rom, 0xff, sizeof(z80rom));
        z80present = 0;
        z80active  = 1;

        z80delay   = 1;
        z80cycle   = 0;
        z80lastcycle = 0;
        z80linecycles = 65*2;

        for (i=0; i<256; i++) {
          bitcount = 0;
          if (i & 0x01) { bitcount++; }
          if (i & 0x02) { bitcount++; }
          if (i & 0x04) { bitcount++; }
          if (i & 0x08) { bitcount++; }
          if (i & 0x10) { bitcount++; }
          if (i & 0x20) { bitcount++; }
          if (i & 0x40) { bitcount++; }
          if (i & 0x80) { bitcount++; }
          flagsszp[i] = ((i == 0)         ? ZFLAG : 0)
                      | ((i & 0x80)       ? SFLAG : 0)
                      | ((bitcount & 0x1) ? 0     : PFLAG);
          flagssz [i] = ((i == 0)         ? ZFLAG : 0)
                      | ((i & 0x80)       ? SFLAG : 0);
        } // for i

        return NULL;    // no error

      } // z80init


/*--------------------------------------*/


      unsigned char *z80loadrom(unsigned int keyboard, unsigned int window) {
        FILE *file;
        unsigned char filepath[260];
        unsigned char oldcwd[260];

        cwdxchgslash(oldcwd, 256);                              // read old directory path
        chdir(inirompath);                                      // set rom directory

        strcpy(filepath, "slotz80.a65");                        // search for source code
        file=fopen(filepath,"rb");
        if (!file) {
          strcpy(filepath, "slotz80.s");
          file=fopen(filepath,"rb");
          if (!file) {
            strcpy(filepath, "slotz80.asm");
            file=fopen(filepath,"rb");
          }
        }
        if (file) {
          fclose(file);                                         // close file again
          stringwritemessage(window,
"!\
EAssemble source code for Z80 slot rom...\r;\
GAssembliere Source Code fr das Z80-Slotrom...\r;\
;");
          screenupdate = 1;
          taskswitch();
          memset(z80rom, 0x00, sizeof(z80rom));
          assembler(z80rom, sizeof(z80rom), filepath);          // assemble source code
          z80present = 1;
        }
        else {
          strcpy(filepath, "z80.rom");;                         // search binary rom file
          file=fopen(filepath,"rb");
          if (file) {
            stringwritemessage(window,
"!\
EReading Z80 slot rom from file...\r;\
GLese Z80-Slotrom aus Datei...\r;\
;");
            screenupdate = 1;
            taskswitch();
            fread(z80rom, sizeof(z80rom), 1, file);             // read binary rom file
            fclose(file);
            z80present = 1;
          }
          else {
            stringwritemessage(window,
"!\
ENo separate Z80 slot rom found\r;\
GKein gesondertes Z80-Slotrom vorhanden\r;\
;");
            screenupdate = 1;
            taskswitch();
            memset(z80rom, 0x00, sizeof(z80rom));               // no special rom
            z80present = 1;                                     // Z80 nevertheless present
          }
        }
        chdir(oldcwd);                                          // restore original directory
//      windowpresskey(keyboard, window);
//      debugger(z80rom, sizeof(z80rom));

        return NULL;    // no error

      } // z80loadrom


/*--------------------------------------*/


      void z80writeregs(unsigned int window) {
        unsigned char message[80];
//                       01234567890123456789012345678901234567890123456789012345678901234567890
        strcpy(message, "A=xx BC=xxxx DE=xxxx HL=xxxx IX=xxxx IY=xxxx PC=xxxx SP=xxxx F=--------");
        hex8tostringf(rega, &message[2]);
        hex8tostringf(regb, &message[8]);
        hex8tostringf(regc, &message[10]);
        hex8tostringf(regd, &message[16]);
        hex8tostringf(rege, &message[18]);
        hex8tostringf(regh, &message[24]);
        hex8tostringf(regl, &message[26]);
        hex8tostringf(regix >> 8,   &message[32]);
        hex8tostringf(regix & 0xff, &message[34]);
        hex8tostringf(regiy >> 8,   &message[40]);
        hex8tostringf(regiy & 0xff, &message[42]);
        hex8tostringf(regpc >> 8,   &message[48]);
        hex8tostringf(regpc & 0xff, &message[50]);
        hex8tostringf(regsp >> 8,   &message[56]);
        hex8tostringf(regsp & 0xff, &message[58]);
        if (regf & SFLAG) {
          message[63] = 'S';
        }
        if (regf & ZFLAG) {
          message[64] = 'Z';
        }
        if (regf & FLAG5) {
          message[65] = '5';
        }
        if (regf & HFLAG) {
          message[66] = 'H';
        }
        if (regf & FLAG3) {
          message[67] = '3';
        }
        if (regf & PFLAG) {
          message[68] = 'P';
        }
        if (regf & NFLAG) {
          message[69] = 'N';
        }
        if (regf & CFLAG) {
          message[70] = 'C';
        }
        stringwrite(window, message);

      } // z80writeregs


/*--------------------------------------*/


      void z80light(unsigned int value) {

        if (value) {
          imagefillbox(window, SLOTX2, 388, SLOTX2+1,389, RGBLGHTON);
        }
        else {
          imagefillbox(window, SLOTX2, 388, SLOTX2+1,389, RGBLGHTOFF);
        }
        screenupdate = 1;

      } // z80light


/*--------------------------------------*/


      unsigned int z80getstate(void) {

        return z80stateflags;

      } // z80getstate


/*--------------------------------------*/


      void z80setstate(unsigned int value) {

        value = value & (CPU_STATEHALT | CPU_STATERESET | CPU_STATETRACE
                       | CPU_STATEIRQ1 | CPU_STATEIRQ2 | CPU_STATENMI1 | CPU_STATENMI2) & (~z80stateflags);


        if (value & CPU_STATEHALT) {
          setmessage("Z80 CPU halted");
          z80light(0);
        }
        if (value & CPU_STATERESET) {
          setmessage("Z80 CPU Reset triggered");
        }
        if (value & CPU_STATETRACE) {
          setmessage("Z80 CPU trace mode set");
        }

/*
        Don't put a message on screen with Dapple, because mouse VBL just generates too many interrupts ...

        if (value & CPU_STATEIRQ1) {
          setmessage("Z80 CPU IRQ triggered");
        }
        if (value & CPU_STATEIRQ2) {
          setmessage("Z80 CPU IRQ triggered");
        }
*/
        if (value & CPU_STATENMI1) {
          setmessage("Z80 CPU NMI1 triggered");
        }
        if (value & CPU_STATENMI2) {
          setmessage("Z80 CPU NMI2 triggered");
        }

        z80stateflags = z80stateflags | value;

      } // z80setstate


/*--------------------------------------*/


      void z80clearstate(unsigned int value) {

        value = value
                & (CPU_STATEHALT | CPU_STATERESET | CPU_STATETRACE | CPU_STATEBPT | CPU_STATEGURU
                 | CPU_STATEIRQ1 | CPU_STATEIRQ2  | CPU_STATENMI1  | CPU_STATENMI2)
                & (z80stateflags);

        if (value & CPU_STATEHALT) {
          setmessage("Z80 CPU started");
          z80light(1);
        }
        if (value & CPU_STATERESET) {
          setmessage("Z80 CPU Reset cleared");
        }
        if (value & CPU_STATEGURU) {    // if it was an illegal opcode ==> reset whole machine
          applereset();
        }

        if (value & CPU_STATEIRQ1) {
          setmessage("Z80 CPU IRQ1 cleared");
        }
        if (value & CPU_STATEIRQ2) {
          setmessage("Z80 CPU IRQ2 cleared");
        }
        if (value & CPU_STATENMI1) {
          setmessage("Z80 CPU NMI cleared");
        }
        if (value & CPU_STATENMI2) {
          setmessage("Z80 CPU NMI cleared");
        }
// don't show message because of too many trace exceptions
/*      if (value & CPU_STATETRACE) {
          setmessage("Z80 CPU trace mode cleared");
        }
*/
        z80stateflags = z80stateflags & (~value);

      } // z80clearstate


/*--------------------------------------*/


      void z80setbreakpoint(unsigned int value) {

        if (value > 0xffff) {
          z80breakpoint = 0xffffffff;
        }
        else {
          z80breakpoint = value;
        }
        z80stateflags = z80stateflags & ~CPU_STATEBPT;

      } // z80setbreakpoint


/*--------------------------------------*/


      unsigned int z80getbreakpoint() {

        return z80breakpoint;

      } // z80getbreakpoint


/*--------------------------------------*/


      void z80clearbreakpoint() {

        z80breakpoint = 0xffffffff;
        z80stateflags = z80stateflags & ~CPU_STATEBPT;

      } // z80clearbreakpoint


/*--------------------------------------*/


#define z80_incpc regpc++

static const unsigned int memtrans[256] = {
        0x1000, 0x1100, 0x1200, 0x1300, 0x1400, 0x1500, 0x1600, 0x1700,
        0x1800, 0x1900, 0x1a00, 0x1b00, 0x1c00, 0x1d00, 0x1e00, 0x1f00,
        0x2000, 0x2100, 0x2200, 0x2300, 0x2300, 0x2500, 0x2600, 0x2700,
        0x2800, 0x2900, 0x2a00, 0x2b00, 0x2c00, 0x2d00, 0x2e00, 0x2f00,
        0x3000, 0x3100, 0x3200, 0x3300, 0x3400, 0x3500, 0x3600, 0x3700,
        0x3800, 0x3900, 0x3a00, 0x3b00, 0x3c00, 0x3d00, 0x3e00, 0x3f00,
        0x4000, 0x4100, 0x4200, 0x4300, 0x4400, 0x4500, 0x4600, 0x4700,
        0x4800, 0x4900, 0x4a00, 0x4b00, 0x4c00, 0x4d00, 0x4e00, 0x4f00,
        0x5000, 0x5100, 0x5200, 0x5300, 0x5400, 0x5500, 0x5600, 0x5700,
        0x5800, 0x5900, 0x5a00, 0x5b00, 0x5c00, 0x5d00, 0x5e00, 0x5f00,
        0x6000, 0x6100, 0x6200, 0x6300, 0x6400, 0x6500, 0x6600, 0x6700,
        0x6800, 0x6900, 0x6a00, 0x6b00, 0x6c00, 0x6d00, 0x6e00, 0x6f00,
        0x7000, 0x7100, 0x7200, 0x7300, 0x7400, 0x7500, 0x7600, 0x7700,
        0x7800, 0x7900, 0x7a00, 0x7b00, 0x7c00, 0x7d00, 0x7e00, 0x7f00,
        0x8000, 0x8100, 0x8200, 0x8300, 0x8400, 0x8500, 0x8600, 0x8700,
        0x8800, 0x8900, 0x8a00, 0x8b00, 0x8c00, 0x8d00, 0x8e00, 0x8f00,
        0x9000, 0x9100, 0x9200, 0x9300, 0x9400, 0x9500, 0x9600, 0x9700,
        0x9800, 0x9900, 0x9a00, 0x9b00, 0x9c00, 0x9d00, 0x9e00, 0x9f00,
        0xa000, 0xa100, 0xa200, 0xa300, 0xa400, 0xa500, 0xa600, 0xa700,
        0xa800, 0xa900, 0xaa00, 0xab00, 0xac00, 0xad00, 0xae00, 0xaf00,
        0xb000, 0xb100, 0xb200, 0xb300, 0xb400, 0xb500, 0xb600, 0xb700,
        0xb800, 0xb900, 0xba00, 0xbb00, 0xbc00, 0xbd00, 0xbe00, 0xbf00,
        0xd000, 0xd100, 0xd200, 0xd300, 0xd400, 0xd500, 0xd600, 0xd700,
        0xd800, 0xd900, 0xda00, 0xdb00, 0xdc00, 0xdd00, 0xde00, 0xdf00,
        0xe000, 0xe100, 0xe200, 0xe300, 0xe400, 0xe500, 0xe600, 0xe700,
        0xe800, 0xe900, 0xea00, 0xeb00, 0xec00, 0xed00, 0xee00, 0xef00,
        0xf000, 0xf100, 0xf200, 0xf300, 0xf400, 0xf500, 0xf600, 0xf700,
        0xf800, 0xf900, 0xfa00, 0xfb00, 0xfc00, 0xfd00, 0xfe00, 0xff00,
        0xc000, 0xc100, 0xc200, 0xc300, 0xc400, 0xc500, 0xc600, 0xc700,
        0xc800, 0xc900, 0xca00, 0xcb00, 0xcc00, 0xcd00, 0xce00, 0xcf00,
        0x0000, 0x0100, 0x0200, 0x0300, 0x0400, 0x0500, 0x0600, 0x0700,
        0x0800, 0x0900, 0x0a00, 0x0b00, 0x0c00, 0x0d00, 0x0e00, 0x0f00
};



// memoryaccess


      unsigned char mread(unsigned int address) {

        return memoryread( memtrans[address >> 8] | (address & 0xff));

      } // mread

      void mwrite(unsigned int address, unsigned char value) {
//      if (address == 0xf3eb) { dvalue3 = regpc; }
        memorywrite( memtrans[address >> 8] | (address & 0xff), value);

      } // mwrite



// stack


      unsigned char z80pullstack(void) {
        register unsigned char value;

        value = mread(regsp);
        regsp = (regsp + 1) & 0xffff;
        return value;

      } // z80pullstack

      void z80pushstack (unsigned int value) {

        regsp = (regsp - 1) & 0xffff;
        mwrite(regsp, value);

      } // z80pushstack



// compare


      void comparea (unsigned char value) {
        register unsigned int w;
        w = rega - value;
        regf = flagssz [ w & 0xff] | (( w & 0x100) >> 8) | NFLAG |
          ((rega ^ w ^ value) & HFLAG) |
          (((value ^ rega) & (value ^ w) & 0x80) >>5);
/*      regf = (regf & ~(SFLAG | ZFLAG | HFLAG | PFLAG | CFLAG))
                | (((rega - value) & 0x80)              ? SFLAG : 0)
                |  ((rega == value)                     ? ZFLAG : 0)
                |  (((rega & 0xf) >= (value & 0xf))     ? 0 : HFLAG)
                |                                         NFLAG
                |  (( rega >= value)                    ? 0 : CFLAG);
*/
      } // comparea


// add


      void adda (unsigned char value) {
        register unsigned int result;

        result = rega + value;
        regf = flagssz [ result & 0xff] | ((result >> 8) & CFLAG) |
          ((rega ^ result ^ value) & HFLAG) |
          (((value ^ rega ^ 0x80) & (value ^ result) >> 5) & PFLAG);
        rega = (unsigned char)result;
      } // adda

      void adca (unsigned char value) {
        register unsigned int result;

        result = rega + value + (regf & CFLAG);
        regf = flagssz [ result & 0xff] | ((result >> 8) & CFLAG) |
          ((rega ^ result ^ value) & HFLAG) |
          (((value ^ rega ^ 0x80) & (value ^ result) >> 5) & PFLAG);
        rega = (unsigned char)result;
      } // adca


// sub

      void suba (unsigned char value) {
        register unsigned int result;

        result = rega - value;
        regf = flagssz [ result & 0xff] | ((result >> 8) & CFLAG) | NFLAG |
          ((rega ^ result ^ value) & HFLAG) |
          (((value ^ rega) & (value ^ result) >> 5) & PFLAG);
        rega = (unsigned char)result;
      } // suba

      void sbca (unsigned char value) {
        register unsigned int result;

        result = rega - value - (regf & 1);
        regf = flagssz [ result & 0xff] | ((result >> 8) & CFLAG) | NFLAG |
          ((rega ^ result ^ value) & HFLAG) |
          (((value ^ rega) & (value ^ result) >> 5) & PFLAG);
        rega = (unsigned char)result;
      } // sbca


// illegal opcodes


      void z80illegal(unsigned int opcode) {
        unsigned char message[80];

        regpc = (regpc - 1) & 0xffff;
        z80stateflags = z80stateflags | CPU_STATEGURU | CPU_STATEHALT;
        z80light(0);

        strcpy(message, "Z80 Illegal instruction $xx hit at address $xxxx");
        hex8tostringf(opcode,    &message[25]);
        hex8tostringf(memtrans[regpc >> 8] >> 8,   &message[44]);
        hex8tostringf(regpc & 0xff, &message[46]);
        setmessage(message);

      } // z80illegal


      void z80halt() {
        unsigned char message[80];

        regpc = (regpc - 1) & 0xffff;
        z80stateflags = z80stateflags | CPU_STATEHALT;
        z80light(0);

        strcpy(message, "Z80 HALT instruction hit at address $xxxx");
        hex8tostringf(memtrans[regpc >> 8] >> 8,   &message[37]);
        hex8tostringf(regpc & 0xff, &message[39]);
        setmessage(message);

      } // z80halt


// opcode emulation


      unsigned int z80line(void) {
        register unsigned char bytebuffer;
        register unsigned int  address;

        do {    // ... while (!z80tracemode)

/* test interrupt signals et al. */
/* The following code is more or less "stolen" from the 6502 code and does not apply to the Z80.
   It's just here for developing the Z80 emulation */
          if (z80stateflags) {
            if (z80stateflags & CPU_STATERESET) {
              z80stateflags = z80stateflags & ~(CPU_STATEHALT | CPU_STATERESET | CPU_STATEGURU | CPU_STATEBPT | CPU_STATEBRK);
              regpc = mread(0xfffe) | (mread(0xffff) << 8);
              z80cycle = z80cycle + 5;
              z80light(1);
              regiff1 = 0;
              regiff2 = 0;
              regpc   = 0x0000;
              regi    = 0;
              z80cycle = z80cycle + 3;
            }
            if (z80stateflags & CPU_STATEHALT) {        // processor still halted?
              screenupdate = 1;
              taskswitch();
              return z80stateflags;
            }
            if (z80stateflags & (CPU_STATENMI1 | CPU_STATENMI1)) {
              z80stateflags = z80stateflags & ~(CPU_STATENMI1 | CPU_STATENMI2);
              regpc   = 0x0066;
              regiff1 = 0;
            }
            else {
              if (z80stateflags & (CPU_STATEIRQ1 | CPU_STATEIRQ2)) {
                if (regiff1) {
                  z80stateflags = z80stateflags & ~(CPU_STATEIRQ1 | CPU_STATEIRQ2);
                  regpc    = 0x0038;            // Mode 2
                  z80cycle = z80cycle + 2;
                } // if (regiff1)
              }
            }
            z80stateflags = z80stateflags &  ~(CPU_STATETRACE | CPU_STATEGURU); // remove these
          } // if (stateflags)

// test for breakpoint
//        if (regpc == z80breakpoint) {
          if (regpc == 0xc5f5) {
            z80stateflags = z80stateflags ^ CPU_STATEBPT;
            if (z80stateflags & CPU_STATEBPT) {
              setmessage("Z80 CPU stopped because breakpoint was reached");
              z80light(0);
              taskswitch();
              return z80stateflags;
            }
          }
          dvalue1 = dvalue2;
          dvalue2 = dvalue3;
          dvalue3 = dvalue4;
          dvalue4 = dvalue5;
          dvalue5 = dvalue6;
          dvalue6 = dvalue7;
          dvalue7 = dvalue8;
          dvalue8 = regpc;


// draw line on screen
          if ((z80cycle - z80lastcycle) >= z80linecycles) {
            z80lastcycle = z80lastcycle + z80linecycles;
            virtline();
            if (!virtrasterline) {
              if (z80traceflag == 2) {
                z80stateflags = z80stateflags | CPU_STATETRACE;
              }
              return z80stateflags;
            }
            z80linecycles       = cpugetlinecycle() << 1;
          }

// interprete next opcode
          bytebuffer    = mread(regpc);
          z80_incpc;
          z80cycle = z80cycle + 4;
          switch (bytebuffer) {
            case 0x00 :         /* NOP */                                       // !
              z80cycle = z80cycle + 4;
              break;
            case 0x01 :         /* LD BC,$nnnn */                               // !
              regc = mread(regpc);
              z80_incpc;
              regb = mread(regpc);
              z80_incpc;
              z80cycle = z80cycle + 10;
              break;
            case 0x02 :         /* LD (BC),A */                                 // !
              mwrite((regb << 8) | regc, rega);
              z80cycle = z80cycle + 7;
              break;
            case 0x03 :         /* INC BC */                                    // !
              regc++;
              if (!regc) {
                regb++;
              }
              z80cycle = z80cycle + 6;
              break;
            case 0x04 :         /* INC B */
              regb++;
              regf = (regf & ~(SFLAG | ZFLAG | HFLAG | PFLAG | NFLAG))
                   |  ((regb &  0x80)           ? SFLAG : 0)
                   |  ((regb == 0x00)           ? ZFLAG : 0)
                   | (((regb  & 0xf) == 0x0)    ? HFLAG : 0)
                   |  ((regb == 0x80)           ? PFLAG : 0);
              z80cycle = z80cycle + 4;
              break;
            case 0x05 :         /* DEC B */
              regb--;
              regf = (regf & ~(SFLAG | ZFLAG | HFLAG | PFLAG))
                   |  ((regb &  0x80)                   ? SFLAG : 0)
                   |  ((regb == 0x00)                   ? ZFLAG : 0)
                   | (((regb  & 0xf) == 0xf)            ? HFLAG : 0)
                   |  ((regb == 0x7f)                   ? PFLAG : 0)
                   |                                      NFLAG;
              z80cycle = z80cycle + 4;
              break;
            case 0x06 :         /* LD B,$nn */                                  // !
              regb = mread(regpc);
              z80_incpc;
              z80cycle = z80cycle + 7;
              break;
            case 0x07 :         /* RLCA */
              if (rega & 0x80) {
                rega = (rega << 1) | 0x01;
                regf = (regf & ~(NFLAG | HFLAG)) | CFLAG;
              }
              else {
                rega = rega << 1;
                regf = regf & ~(NFLAG | HFLAG);
              }
              z80cycle = z80cycle + 4;
              break;
            case 0x08 :         /* EX AF,AF' */
              bytebuffer = rega;
              rega = rega2;
              rega2 = bytebuffer;
              bytebuffer = regf;
              regf = regf2;
              regf2 = bytebuffer;
              z80cycle = z80cycle + 4;
              break;
            case 0x09 :         /* ADD HL,BC */                                 // !
              regf = regf & ~(HFLAG | NFLAG | CFLAG);
              address = (regh << 8) | regl;
              if (((address & 0xfff) + (((regb << 8) | regc) & 0xfff)) > 0xfff) {
                regf = regf | HFLAG;
              }
              address = address + ((regb << 8) | regc);
              if (address > 0xffff) {
                regf = regf | CFLAG;
              }
              regl = (unsigned char)address;
              regh = (unsigned char)(address >> 8);
              z80cycle = z80cycle + 11;
              break;
            case 0x0a :         /* LD A,(BC) */                                 // !
              rega = mread((regb << 8) | regc);
              z80cycle = z80cycle + 7;
              break;
            case 0x0b :         /* DEC BC */
              if (!regc) {
                regb--;
              }
              regc--;
              z80cycle = z80cycle + 6;
              break;
            case 0x0c :         /* INC C */
              regc++;
              regf = (regf & ~(SFLAG | ZFLAG | HFLAG | PFLAG | NFLAG))
                   |  ((regc &  0x80)           ? SFLAG : 0)
                   |  ((regc == 0x00)           ? ZFLAG : 0)
                   | (((regc  & 0xf) == 0x0)    ? HFLAG : 0)
                   |  ((regc == 0x80)           ? PFLAG : 0);
              z80cycle = z80cycle + 4;
              break;
            case 0x0d :         /* DEC C */
              regc--;
              regf = (regf & ~(SFLAG | ZFLAG | HFLAG | PFLAG))
                   |  ((regc &  0x80)                   ? SFLAG : 0)
                   |  ((regc == 0x00)                   ? ZFLAG : 0)
                   | (((regc  & 0xf) == 0xf)            ? HFLAG : 0)
                   |  ((regc == 0x7f)                   ? PFLAG : 0)
                   |                                      NFLAG;
              z80cycle = z80cycle + 4;
              break;
            case 0x0e :         /* LD C,$nn */                                  // !
              regc = mread(regpc);
              z80_incpc;
              z80cycle = z80cycle + 7;
              break;
            case 0x0f :         /* RRCA */
              if (rega & 0x01) {
                rega = (rega >> 1) | 0x80;
                regf = (regf & ~(NFLAG | HFLAG)) | CFLAG;
              }
              else {
                rega = rega >> 1;
                regf = regf & ~(NFLAG | HFLAG);
              }
              z80cycle = z80cycle + 4;
              break;
            case 0x10 :         /* DJNZ $rr */                                  // !
              regb--;
              if (regb) {
                bytebuffer = mread(regpc);
                if ( bytebuffer > 0x7f ) {
                  regpc = (regpc - 255 + bytebuffer) & 0xffff;
                }
                else {
                  regpc = (regpc + 1 + bytebuffer) & 0xffff;
                }
                z80cycle = z80cycle + 13;
              }
              else {
                z80_incpc;
                z80cycle = z80cycle + 8;
              }
              break;
            case 0x11 :         /* LD DE,$nnnn */                               // !
              rege = mread(regpc);
              z80_incpc;
              regd = mread(regpc);
              z80_incpc;
              z80cycle = z80cycle + 10;
              break;
            case 0x12 :         /* LD (DE),A */                                 // !
              mwrite((regd << 8) | rege, rega);
              z80cycle = z80cycle + 7;
              break;
            case 0x13 :         /* INC DE */                                    // !
              rege++;
              if (!rege) {
                regd++;
              }
              z80cycle = z80cycle + 6;
              break;
            case 0x14 :         /* INC D */
              regd++;
              regf = (regf & ~(SFLAG | ZFLAG | HFLAG | PFLAG | NFLAG))
                   |  ((regd &  0x80)           ? SFLAG : 0)
                   |  ((regd == 0x00)           ? ZFLAG : 0)
                   | (((regd  & 0xf) == 0x0)    ? HFLAG : 0)
                   |  ((regd == 0x80)           ? PFLAG : 0);
              z80cycle = z80cycle + 4;
              break;
            case 0x15 :         /* DEC D */
              regd--;
              regf = (regf & ~(SFLAG | ZFLAG | HFLAG | PFLAG))
                   |  ((regd &  0x80)                   ? SFLAG : 0)
                   |  ((regd == 0x00)                   ? ZFLAG : 0)
                   | (((regd  & 0xf) == 0xf)            ? HFLAG : 0)
                   |  ((regd == 0x7f)                   ? PFLAG : 0)
                   |                                      NFLAG;
              z80cycle = z80cycle + 4;
              break;
            case 0x16 :         /* LD D,$nn */                                  // !
              regd = mread(regpc);
              z80_incpc;
              z80cycle = z80cycle + 7;
              break;
            case 0x17 :         /* RLA */
              if (regf & CFLAG) {
                if (rega & 0x80) {
                  regf = (regf | CFLAG) & ~(HFLAG | NFLAG);
                }
                else {
                  regf = (regf) & ~(CFLAG | HFLAG | NFLAG);
                }
                rega = (rega << 1) | 0x01;
              }
              else {
                if (rega & 0x80) {
                  regf = (regf | CFLAG) & ~(HFLAG | NFLAG);
                }
                else {
                  regf = (regf) & ~(CFLAG | HFLAG | NFLAG);
                }
                rega = rega << 1;
              }
              z80cycle = z80cycle + 4;
              break;
            case 0x18 :         /* JR $NN */                                    // !
              bytebuffer = mread(regpc);
              if ( bytebuffer > 0x7f ) {
                regpc   = (regpc - 255 + bytebuffer) & 0xffff;
              }
              else {
                regpc   = (regpc + 1 + bytebuffer) & 0xffff;
              }
              z80cycle = z80cycle + 12;
              break;
            case 0x19 :         /* ADD HL,DE */                                 // !
              regf = regf & ~(HFLAG | NFLAG | CFLAG);
              address = (regh << 8) | regl;
              if (((address & 0xfff) + (((regd << 8) | rege) & 0xfff)) > 0xfff) {
                regf = regf | HFLAG;
              }
              address = address + ((regd << 8) | rege);
              if (address > 0xffff) {
                regf = regf | CFLAG;
              }
              regl = (unsigned char)address;
              regh = (unsigned char)(address >> 8);
              z80cycle = z80cycle + 11;
              break;
            case 0x1a :         /* LD A,(DE) */                                 // !
              rega = mread((regd << 8) | rege);
              z80cycle = z80cycle + 7;
              break;
            case 0x1b :         /* DEC DE */
              if (!rege) {
                regd--;
              }
              rege--;
              z80cycle = z80cycle + 6;
              break;
            case 0x1c :         /* INC E */
              rege++;
              regf = (regf & ~(SFLAG | ZFLAG | HFLAG | PFLAG | NFLAG))
                   |  ((rege &  0x80)           ? SFLAG : 0)
                   |  ((rege == 0x00)           ? ZFLAG : 0)
                   | (((rege  & 0xf) == 0x0)    ? HFLAG : 0)
                   |  ((rege == 0x80)           ? PFLAG : 0);
              z80cycle = z80cycle + 4;
              break;
            case 0x1d :         /* DEC E */
              rege--;
              regf = (regf & ~(SFLAG | ZFLAG | HFLAG | PFLAG))
                   |  ((rege &  0x80)                   ? SFLAG : 0)
                   |  ((rege == 0x00)                   ? ZFLAG : 0)
                   | (((rege  & 0xf) == 0xf)            ? HFLAG : 0)
                   |  ((rege == 0x7f)                   ? PFLAG : 0)
                   |                                      NFLAG;
              z80cycle = z80cycle + 4;
              break;
            case 0x1e :         /* LD E,$nn */                                  // !
              rege = mread(regpc);
              z80_incpc;
              z80cycle = z80cycle + 7;
              break;
            case 0x1f :         /* RRA */
              if (regf & CFLAG) {
                if (rega & 0x01) {
                  regf = (regf | CFLAG) & ~(HFLAG | NFLAG);
                }
                else {
                  regf = (regf) & ~(CFLAG | HFLAG | NFLAG);
                }
                rega = (rega >> 1) | 0x80;
              }
              else {
                if (rega & 0x01) {
                  regf = (regf | CFLAG) & ~(HFLAG | NFLAG);
                }
                else {
                  regf = (regf) & ~(CFLAG | HFLAG | NFLAG);
                }
                rega = rega >> 1;
              }
              z80cycle = z80cycle + 4;
              break;
            case 0x20 :         /* JR NZ,$NN */                                 // !
              if (!(regf & ZFLAG)) {
                bytebuffer = mread(regpc);
                if ( bytebuffer > 0x7f ) {
                  regpc = (regpc - 255 + bytebuffer) & 0xffff;
                }
                else {
                  regpc = (regpc + 1 + bytebuffer) & 0xffff;
                }
                z80cycle = z80cycle + 12;
              }
              else {
                z80_incpc;
                z80cycle = z80cycle + 7;
              }
              break;
            case 0x21 :         /* LD HL,$nnnn */                               // !
              regl = mread(regpc);
              z80_incpc;
              regh = mread(regpc);
              z80_incpc;
              z80cycle = z80cycle + 10;
              break;
            case 0x22 :         /* LD ($nnnn),HL */                             // !
              bytebuffer = mread(regpc);
              z80_incpc;
              address = (mread(regpc) << 8) | bytebuffer;
              z80_incpc;
              mwrite(address, regl);
              mwrite((address + 1) & 0xffff, regh);
              z80cycle = z80cycle + 16;
              break;
            case 0x23 :         /* INC HL */                                    // !
              regl++;
              if (!regl) {
                regh++;
              }
              z80cycle = z80cycle + 6;
              break;
            case 0x24 :         /* INC H */
              regh++;
              regf = (regf & ~(SFLAG | ZFLAG | HFLAG | PFLAG | NFLAG))
                   |  ((regh &  0x80)           ? SFLAG : 0)
                   |  ((regh == 0x00)           ? ZFLAG : 0)
                   | (((regh  & 0xf) == 0x0)    ? HFLAG : 0)
                   |  ((regh == 0x80)           ? PFLAG : 0);
              z80cycle = z80cycle + 4;
              break;
            case 0x25 :         /* DEC H */
              regh--;
              regf = (regf & ~(SFLAG | ZFLAG | HFLAG | PFLAG))
                   |  ((regh &  0x80)                   ? SFLAG : 0)
                   |  ((regh == 0x00)                   ? ZFLAG : 0)
                   | (((regh  & 0xf) == 0xf)            ? HFLAG : 0)
                   |  ((regh == 0x7f)                   ? PFLAG : 0)
                   |                                      NFLAG;
              z80cycle = z80cycle + 4;
              break;
            case 0x26 :         /* LD H,$nn */                                  // !
              regh = mread(regpc);
              z80_incpc;
              z80cycle = z80cycle + 7;
              break;
                                /* DAA still missing */
            case 0x28 :         /* JR Z,$NN */                                  // !
              if (regf & ZFLAG) {
                bytebuffer = mread(regpc);
                if ( bytebuffer > 0x7f ) {
                  regpc = (regpc - 255 + bytebuffer) & 0xffff;
                }
                else {
                  regpc = (regpc + 1 + bytebuffer) & 0xffff;
                }
                z80cycle = z80cycle + 12;
              }
              else {
                z80_incpc;
                z80cycle = z80cycle + 7;
              }
              break;
            case 0x29 :         /* ADD HL,HL */                                 // !
              regf = regf & ~(HFLAG | NFLAG | CFLAG);
              address = (regh << 8) | regl;
              if (((address & 0xfff) + (address & 0xfff)) > 0xfff) {
                regf = regf | HFLAG;
              }
              address = address + address;
              if (address > 0xffff) {
                regf = regf | CFLAG;
              }
              regl = (unsigned char)address;
              regh = (unsigned char)(address >> 8);
              z80cycle = z80cycle + 11;
              break;
            case 0x2a :         /* LD HL,($nnnn) */                             // !
              bytebuffer = mread(regpc);
              z80_incpc;
              address = (mread(regpc) << 8) | bytebuffer;
              z80_incpc;
              regl = mread(address);
              regh = mread((address + 1) & 0xffff);
              z80cycle = z80cycle + 20;
              break;
            case 0x2b :         /* DEC HL */
              if (!regl) {
                regh--;
              }
              regl--;
              z80cycle = z80cycle + 6;
              break;
            case 0x2c :         /* INC L */
              regl++;
              regf = (regf & ~(SFLAG | ZFLAG | HFLAG | PFLAG | NFLAG))
                   |  ((regl &  0x80)           ? SFLAG : 0)
                   |  ((regl == 0x00)           ? ZFLAG : 0)
                   | (((regl  & 0xf) == 0x0)    ? HFLAG : 0)
                   |  ((regl == 0x80)           ? PFLAG : 0);
              z80cycle = z80cycle + 4;
              break;
            case 0x2d :         /* DEC L */
              regl--;
              regf = (regf & ~(SFLAG | ZFLAG | HFLAG | PFLAG))
                   |  ((regl &  0x80)                   ? SFLAG : 0)
                   |  ((regl == 0x00)                   ? ZFLAG : 0)
                   | (((regl  & 0xf) == 0xf)            ? HFLAG : 0)
                   |  ((regl == 0x7f)                   ? PFLAG : 0)
                   |                                      NFLAG;
              z80cycle = z80cycle + 4;
              break;
            case 0x2e :         /* LD L,$nn */                                  // !
              regl = mread(regpc);
              z80_incpc;
              z80cycle = z80cycle + 7;
              break;
            case 0x2f :         /* CPL */
              rega = ~rega;
              regf = regf | (HFLAG | NFLAG);
              z80cycle = z80cycle + 4;
              break;
            case 0x30 :         /* JR NC,$NN */                                 // !
              if (!(regf & CFLAG)) {
                bytebuffer = mread(regpc);
                if ( bytebuffer > 0x7f ) {
                  regpc = (regpc - 255 + bytebuffer) & 0xffff;
                }
                else {
                  regpc = (regpc + 1 + bytebuffer) & 0xffff;
                }
                z80cycle = z80cycle + 12;
              }
              else {
                z80_incpc;
                z80cycle = z80cycle + 7;
              }
              break;
            case 0x31 :         /* LD SP,$nnnn */                               // !
              bytebuffer = mread(regpc);
              z80_incpc;
              regsp = (mread(regpc) << 8) | bytebuffer;
              z80_incpc;
              z80cycle = z80cycle + 10;
              break;
            case 0x32 :         /* LD ($nnnn),A */                              // !
              bytebuffer = mread(regpc);
              z80_incpc;
              address = (mread(regpc) << 8) | bytebuffer;
              z80_incpc;
              mwrite(address, rega);
              z80cycle = z80cycle + 13;
              break;
            case 0x33 :         /* INC SP */
              regsp = (regsp + 1) & 0xffff;
              z80cycle = z80cycle + 6;
              break;
            case 0x34 :         /* INC (HL) */
              address = (regh << 8) | regl;
              bytebuffer = mread(address);
              bytebuffer++;
              regf = (regf & ~(SFLAG | ZFLAG | HFLAG | PFLAG | NFLAG))
                   |  ((bytebuffer &  0x80)             ? SFLAG : 0)
                   |  ((bytebuffer == 0x00)             ? ZFLAG : 0)
                   | (((bytebuffer  & 0xf) == 0x0)      ? HFLAG : 0)
                   |  ((bytebuffer == 0x80)             ? PFLAG : 0);
              mwrite(address, bytebuffer);
              z80cycle = z80cycle + 11;
              break;
            case 0x35 :         /* DEC (HL) */                                  // !
              address = (regh << 8) | regl;
              bytebuffer = mread(address);
              bytebuffer--;
              regf = (regf & ~(SFLAG | ZFLAG | HFLAG | PFLAG | NFLAG))
                   |  ((bytebuffer &  0x80)             ? SFLAG : 0)
                   |  ((bytebuffer == 0x00)             ? ZFLAG : 0)
                   | (((bytebuffer  & 0xf) == 0xf)      ? HFLAG : 0)
                   |  ((bytebuffer == 0x7f)             ? PFLAG : 0);
              mwrite(address, bytebuffer);
              z80cycle = z80cycle + 11;
              break;
            case 0x36 :         /* LD (HL),$nn */                               // !
              mwrite((regh << 8) | regl, mread(regpc));
              z80_incpc;
              z80cycle = z80cycle + 10;
              break;
            case 0x37 :         /* SCF */                                       // !
              regf = regf | CFLAG;
              z80cycle = z80cycle + 4;
              break;
            case 0x38 :         /* JR C,$NN */                                  // !
              if (regf & CFLAG) {
                bytebuffer = mread(regpc);
                if ( bytebuffer > 0x7f ) {
                  regpc = (regpc - 255 + bytebuffer) & 0xffff;
                }
                else {
                  regpc = (regpc + 1 + bytebuffer) & 0xffff;
                }
                z80cycle = z80cycle + 12;
              }
              else {
                z80_incpc;
                z80cycle = z80cycle + 7;
              }
              break;
            case 0x39 :         /* ADD HL,SP */                                 // !
              regf = regf & ~(HFLAG | NFLAG | CFLAG);
              address = (regh << 8) | regl;
              if (((address & 0xfff) + (regsp & 0xfff)) > 0xfff) {
                regf = regf | HFLAG;
              }
              address = address + regsp;
              if (address > 0xffff) {
                regf = regf | CFLAG;
              }
              regl = (unsigned char)address;
              regh = (unsigned char)(address >> 8);
              z80cycle = z80cycle + 11;
              break;
            case 0x3a :         /* LD A,($nnnn) */                              // !
              bytebuffer = mread(regpc);
              z80_incpc;
              address = (mread(regpc) << 8) | bytebuffer;
              z80_incpc;
              rega = mread(address);
              z80cycle = z80cycle + 13;
              break;
            case 0x3b :         /* DEC SP */
              regsp = (regsp - 1) & 0xffff;
              z80cycle = z80cycle + 6;
              break;
            case 0x3c :         /* INC A */
              rega++;
              regf = (regf & ~(SFLAG | ZFLAG | HFLAG | PFLAG | NFLAG))
                   |  ((rega &  0x80)           ? SFLAG : 0)
                   |  ((rega == 0x00)           ? ZFLAG : 0)
                   | (((rega  & 0xf) == 0x0)    ? HFLAG : 0)
                   |  ((rega == 0x80)           ? PFLAG : 0);
              z80cycle = z80cycle + 4;
              break;
            case 0x3d :         /* DEC A */
              rega--;
              regf = (regf & ~(SFLAG | ZFLAG | HFLAG | PFLAG))
                   | ((rega &  0x80)                    ? SFLAG : 0)
                   | ((rega == 0x00)                    ? ZFLAG : 0)
                   | (((rega  & 0xf) == 0xf)            ? HFLAG : 0)
                   | ((rega == 0x7f)                    ? PFLAG : 0)
                   |                                      NFLAG;
              z80cycle = z80cycle + 4;
              break;
            case 0x3e :         /* LD A,$nn */                          // !
              rega = mread(regpc);
              z80_incpc;
              z80cycle = z80cycle + 7;
              break;
            case 0x3f :         /* CCF */                               // !
              regf = regf ^ CFLAG;
              z80cycle = z80cycle + 4;
              break;
            case 0x40 :         /* LD B,B */                            // !
              z80cycle = z80cycle + 4;
              break;
            case 0x41 :         /* LD B,C */                            // !
              regb = regc;
              z80cycle = z80cycle + 4;
              break;
            case 0x42 :         /* LD B,D */                            // !
              regb = regd;
              z80cycle = z80cycle + 4;
              break;
            case 0x43 :         /* LD B,E */                            // !
              regb = rege;
              z80cycle = z80cycle + 4;
              break;
            case 0x44 :         /* LD B,H */                            // !
              regb = regh;
              z80cycle = z80cycle + 4;
              break;
            case 0x45 :         /* LD B,L */                            // !
              regb = regl;
              z80cycle = z80cycle + 4;
              break;
            case 0x46 :         /* LD B,(HL) */                         // !
              regb = mread((regh << 8) | regl);
              z80cycle = z80cycle + 7;
              break;
            case 0x47 :         /* LD B,A */                            // !
              regb = rega;
              z80cycle = z80cycle + 4;
              break;
            case 0x48 :         /* LD C,B */                            // !
              regc = regb;
              z80cycle = z80cycle + 4;
              break;
            case 0x49 :         /* LD C,C */                            // !
              z80cycle = z80cycle + 4;
              break;
            case 0x4a :         /* LD C,D */                            // !
              regc = regd;
              z80cycle = z80cycle + 4;
              break;
            case 0x4b :         /* LD C,E */                            // !
              regc = rege;
              z80cycle = z80cycle + 4;
              break;
            case 0x4c :         /* LD C,H */                            // !
              regc = regh;
              z80cycle = z80cycle + 4;
              break;
            case 0x4d :         /* LD C,L */                            // !
              regc = regl;
              z80cycle = z80cycle + 4;
              break;
            case 0x4e :         /* LD C,(HL) */                         // !
              regc = mread((regh << 8) | regl);
              z80cycle = z80cycle + 7;
              break;
            case 0x4f :         /* LD C,A */                            // !
              regc = rega;
              z80cycle = z80cycle + 4;
              break;
            case 0x50 :         /* LD D,B */                            // !
              regd = regb;
              z80cycle = z80cycle + 4;
              break;
            case 0x51 :         /* LD D,C */                            // !
              regd = regc;
              z80cycle = z80cycle + 4;
              break;
            case 0x52 :         /* LD D,D */                            // !
              z80cycle = z80cycle + 4;
              break;
            case 0x53 :         /* LD D,E */                            // !
              regd = rege;
              z80cycle = z80cycle + 4;
              break;
            case 0x54 :         /* LD D,H */                            // !
              regd = regh;
              z80cycle = z80cycle + 4;
              break;
            case 0x55 :         /* LD D,L */                            // !
              regd = regl;
              z80cycle = z80cycle + 4;
              break;
            case 0x56 :         /* LD D,(HL) */                         // !
              regd = mread( (regh << 8) | regl);
              z80cycle = z80cycle + 7;
              break;
            case 0x57 :         /* LD D,A */                            // !
              regd = rega;
              z80cycle = z80cycle + 4;
              break;
            case 0x58 :         /* LD E,B */                            // !
              rege = regb;
              z80cycle = z80cycle + 4;
              break;
            case 0x59 :         /* LD E,C */                            // !
              rege = regc;
              z80cycle = z80cycle + 4;
              break;
            case 0x5a :         /* LD E,D */                            // !
              rege = regd;
              z80cycle = z80cycle + 4;
              break;
            case 0x5b :         /* LD E,E */                            // !
              z80cycle = z80cycle + 4;
              break;
            case 0x5c :         /* LD E,H */                            // !
              rege = regh;
              z80cycle = z80cycle + 4;
              break;
            case 0x5d :         /* LD E,L */                            // !
              rege = regl;
              z80cycle = z80cycle + 4;
              break;
            case 0x5e :         /* LD E,(HL) */                         // !
              rege = mread( (regh << 8) | regl);
              z80cycle = z80cycle + 7;
              break;
            case 0x5f :         /* LD E,A */                            // !
              rege = rega;
              z80cycle = z80cycle + 4;
              break;
            case 0x60 :         /* LD H,B */                            // !
              regh = regb;
              z80cycle = z80cycle + 4;
              break;
            case 0x61 :         /* LD H,C */                            // !
              regh = regc;
              z80cycle = z80cycle + 4;
              break;
            case 0x62 :         /* LD H,D */                            // !
              regh = regd;
              z80cycle = z80cycle + 4;
              break;
            case 0x63 :         /* LD H,E */                            // !
              regh = rege;
              z80cycle = z80cycle + 4;
              break;
            case 0x64 :         /* LD H,H */                            // !
              z80cycle = z80cycle + 4;
              break;
            case 0x65 :         /* LD H,L */                            // !
              regh = regl;
              z80cycle = z80cycle + 4;
              break;
            case 0x66 :         /* LD H,(HL) */                         // !
              regh = mread((regh << 8) | regl);
              z80cycle = z80cycle + 7;
              break;
            case 0x67 :         /* LD H,A */                            // !
              regh = rega;
              z80cycle = z80cycle + 4;
              break;
            case 0x68 :         /* LD L,B */                            // !
              regl = regb;
              z80cycle = z80cycle + 4;
              break;
            case 0x69 :         /* LD L,C */                            // !
              regl = regc;
              z80cycle = z80cycle + 4;
              break;
            case 0x6a :         /* LD L,D */                            // !
              regl = regd;
              z80cycle = z80cycle + 4;
              break;
            case 0x6b :         /* LD L,E */                            // !
              regl = rege;
              z80cycle = z80cycle + 4;
              break;
            case 0x6c :         /* LD L,H */                            // !
              regl = regh;
              z80cycle = z80cycle + 4;
              break;
            case 0x6d :         /* LD L,L */                            // !
              z80cycle = z80cycle + 4;
              break;
            case 0x6e :         /* LD L,(HL) */                         // !
              regl = mread( (regh << 8) | regl);
              z80cycle = z80cycle + 7;
              break;
            case 0x6f :         /* LD L,A */                            // !
              regl = rega;
              z80cycle = z80cycle + 4;
              break;
            case 0x70 :         /* LD (HL),B */                         // !
              mwrite((regh << 8) | regl, regb);
              z80cycle = z80cycle + 7;
              break;
            case 0x71 :         /* LD (HL),C */                         // !
              mwrite((regh << 8) | regl, regc);
              z80cycle = z80cycle + 7;
              break;
            case 0x72 :         /* LD (HL),D */                         // !
              mwrite((regh << 8) | regl, regd);
              z80cycle = z80cycle + 7;
              break;
            case 0x73 :         /* LD (HL),E */                         // !
              mwrite((regh << 8) | regl, rege);
              z80cycle = z80cycle + 7;
              break;
            case 0x74 :         /* LD (HL),H */                         // !
              mwrite((regh << 8) | regl, regh);
              z80cycle = z80cycle + 7;
              break;
            case 0x75 :         /* LD (HL),L */                         // !
              mwrite((regh << 8) | regl, regl);
              z80cycle = z80cycle + 7;
              break;
            case 0x76 :         /* HALT */
              z80halt();
              z80cycle = z80cycle + 4;
              return z80stateflags;
            case 0x77 :         /* LD (HL),A */                         // !
              mwrite((regh << 8) | regl, rega);
              z80cycle = z80cycle + 7;
              break;
            case 0x78 :         /* LD A,B */                            // !
              rega = regb;
              z80cycle = z80cycle + 4;
              break;
            case 0x79 :         /* LD A,C */                            // !
              rega = regc;
              z80cycle = z80cycle + 4;
              break;
            case 0x7a :         /* LD A,D */                            // !
              rega = regd;
              z80cycle = z80cycle + 4;
              break;
            case 0x7b :         /* LD A,E */                            // !
              rega = rege;
              z80cycle = z80cycle + 4;
              break;
            case 0x7c :         /* LD A,H */                            // !
              rega = regh;
              z80cycle = z80cycle + 4;
              break;
            case 0x7d :         /* LD A,L */                            // !
              rega = regl;
              z80cycle = z80cycle + 4;
              break;
            case 0x7e :         /* LD A,(HL) */                         // !
              rega = mread((regh << 8) | regl);
              z80cycle = z80cycle + 7;
              break;
            case 0x7f :         /* LD A,A */                            // !
              z80cycle = z80cycle + 4;
              break;
            case 0x80 :         /* ADD B */
              adda(regb);
              z80cycle = z80cycle + 4;
              break;
            case 0x81 :         /* ADD C */
              adda(regc);
              z80cycle = z80cycle + 4;
              break;
            case 0x82 :         /* ADD D */
              adda(regd);
              z80cycle = z80cycle + 4;
              break;
            case 0x83 :         /* ADD E */
              adda(rege);
              z80cycle = z80cycle + 4;
              break;
            case 0x84 :         /* ADD H */
              adda(regh);
              z80cycle = z80cycle + 4;
              break;
            case 0x85 :         /* ADD L */
              adda(regl);
              z80cycle = z80cycle + 4;
              break;
            case 0x86 :         /* ADD (HL) */
              adda(mread((regh << 8) | regl));
              z80cycle = z80cycle + 7;
              break;
            case 0x87 :         /* ADD A */
              adda(rega);
              z80cycle = z80cycle + 4;
              break;
            case 0x88 :         /* ADC B */
              adca(regb);
              z80cycle = z80cycle + 4;
              break;
            case 0x89 :         /* ADC C */
              adca(regc);
              z80cycle = z80cycle + 4;
              break;
            case 0x8a :         /* ADC D */
              adca(regd);
              z80cycle = z80cycle + 4;
              break;
            case 0x8b :         /* ADC E */
              adca(rege);
              z80cycle = z80cycle + 4;
              break;
            case 0x8c :         /* ADC H */
              adca(regh);
              z80cycle = z80cycle + 4;
              break;
            case 0x8d :         /* ADC L */
              adca(regl);
              z80cycle = z80cycle + 4;
              break;
            case 0x8e :         /* ADC A */
              adca(rega);
              z80cycle = z80cycle + 4;
              break;
            case 0x8f :         /* ADC A */
              adca(rega);
              z80cycle = z80cycle + 4;
              break;
            case 0x90 :         /* SUB B */
              suba(regb);
              z80cycle = z80cycle + 4;
              break;
            case 0x91 :         /* SUB C */
              suba(regc);
              z80cycle = z80cycle + 4;
              break;
            case 0x92 :         /* SUB D */
              suba(regd);
              z80cycle = z80cycle + 4;
              break;
            case 0x93 :         /* SUB E */
              suba(rege);
              z80cycle = z80cycle + 4;
              break;
            case 0x94 :         /* SUB H */
              suba(regh);
              z80cycle = z80cycle + 4;
              break;
            case 0x95 :         /* SUB L */
              suba(regl);
              z80cycle = z80cycle + 4;
              break;
            case 0x96 :         /* SUB (HL) */
              suba(mread((regh << 8) | regl));
              z80cycle = z80cycle + 7;
              break;
            case 0x97 :         /* SUB A */
              suba(rega);
              z80cycle = z80cycle + 4;
              break;
            case 0x98 :         /* SBC B */
              sbca(regb);
              z80cycle = z80cycle + 4;
              break;
            case 0x99 :         /* SBC C */
              sbca(regc);
              z80cycle = z80cycle + 4;
              break;
            case 0x9a :         /* SBC D */
              sbca(regd);
              z80cycle = z80cycle + 4;
              break;
            case 0x9b :         /* SBC E */
              sbca(rege);
              z80cycle = z80cycle + 4;
              break;
            case 0x9c :         /* SBC H */
              sbca(regh);
              z80cycle = z80cycle + 4;
              break;
            case 0x9d :         /* SBC L */
              sbca(regl);
              z80cycle = z80cycle + 4;
              break;
            case 0x9e :         /* SBC (HL) */
              sbca(mread((regh << 8) | regl));
              z80cycle = z80cycle + 7;
              break;
            case 0x9f :         /* SBC A */
              sbca(rega);
              z80cycle = z80cycle + 4;
              break;
            case 0xa0 :         /* AND B */
              rega = rega & regb;
              regf = (regf & (FLAG3 | FLAG5)) | flagsszp[rega] | HFLAG;
              z80cycle = z80cycle + 4;
              break;
            case 0xa1 :         /* AND C */
              rega = rega & regc;
              regf = (regf & (FLAG3 | FLAG5)) | flagsszp[rega] | HFLAG;
              z80cycle = z80cycle + 4;
              break;
            case 0xa2 :         /* AND D */
              rega = rega & regd;
              regf = (regf & (FLAG3 | FLAG5)) | flagsszp[rega] | HFLAG;
              z80cycle = z80cycle + 4;
              break;
            case 0xa3 :         /* AND E */
              rega = rega & rege;
              regf = (regf & (FLAG3 | FLAG5)) | flagsszp[rega] | HFLAG;
              z80cycle = z80cycle + 4;
              break;
            case 0xa4 :         /* AND H */
              rega = rega & regh;
              regf = (regf & (FLAG3 | FLAG5)) | flagsszp[rega] | HFLAG;
              z80cycle = z80cycle + 4;
              break;
            case 0xa5 :         /* AND L */
              rega = rega & regl;
              regf = (regf & (FLAG3 | FLAG5)) | flagsszp[rega] | HFLAG;
              z80cycle = z80cycle + 4;
              break;
            case 0xa6 :         /* AND (HL) */
              rega = rega & mread((regh << 8) | regl);
              regf = (regf & (FLAG3 | FLAG5)) | flagsszp[rega] | HFLAG;
              z80cycle = z80cycle + 7;
              break;
            case 0xa7 :         /* AND A */
              regf = (regf & (FLAG3 | FLAG5)) | flagsszp[rega] | HFLAG;
              z80cycle = z80cycle + 4;
              break;
            case 0xa8 :         /* XOR B */
              rega = rega ^ regb;
              regf = (regf & (FLAG3 | FLAG5)) | flagsszp[rega];
              z80cycle = z80cycle + 4;
              break;
            case 0xa9 :         /* XOR C */
              rega = rega ^ regc;
              regf = (regf & (FLAG3 | FLAG5)) | flagsszp[rega];
              z80cycle = z80cycle + 4;
              break;
            case 0xaa :         /* XOR D */
              rega = rega ^ regd;
              regf = (regf & (FLAG3 | FLAG5)) | flagsszp[rega];
              z80cycle = z80cycle + 4;
              break;
            case 0xab :         /* XOR E */
              rega = rega ^ rege;
              regf = (regf & (FLAG3 | FLAG5)) | flagsszp[rega];
              z80cycle = z80cycle + 4;
              break;
            case 0xac :         /* XOR H */
              rega = rega ^ regh;
              regf = (regf & (FLAG3 | FLAG5)) | flagsszp[rega];
              z80cycle = z80cycle + 4;
              break;
            case 0xad :         /* XOR L */
              rega = rega ^ regl;
              regf = (regf & (FLAG3 | FLAG5)) | flagsszp[rega];
              z80cycle = z80cycle + 4;
              break;
            case 0xae :         /* XOR (HL) */
              rega = rega ^ mread((regh << 8) | regl);
              regf = (regf & (FLAG3 | FLAG5)) | flagsszp[rega];
              z80cycle = z80cycle + 7;
              break;
            case 0xaf :         /* XOR A */
              rega = 0x00;
              regf = (regf & (FLAG3 | FLAG5)) | (ZFLAG | PFLAG);
              z80cycle = z80cycle + 4;
              break;
            case 0xb0 :         /* OR B */
              rega = rega | regb;
              regf = (regf & (FLAG3 | FLAG5)) | flagsszp[rega];
              z80cycle = z80cycle + 4;
              break;
            case 0xb1 :         /* OR C */
              rega = rega | regc;
              regf = (regf & (FLAG3 | FLAG5)) | flagsszp[rega];
              z80cycle = z80cycle + 4;
              break;
            case 0xb2 :         /* OR D */
              rega = rega | regd;
              regf = (regf & (FLAG3 | FLAG5)) | flagsszp[rega];
              z80cycle = z80cycle + 4;
              break;
            case 0xb3 :         /* OR E */
              rega = rega | rege;
              regf = (regf & (FLAG3 | FLAG5)) | flagsszp[rega];
              z80cycle = z80cycle + 4;
              break;
            case 0xb4 :         /* OR H */
              rega = rega | regh;
              regf = (regf & (FLAG3 | FLAG5)) | flagsszp[rega];
              z80cycle = z80cycle + 4;
              break;
            case 0xb5 :         /* OR L */
              rega = rega | regl;
              regf = (regf & (FLAG3 | FLAG5)) | flagsszp[rega];
              z80cycle = z80cycle + 4;
              break;
            case 0xb6 :         /* OR (HL) */
              rega = rega | mread((regh << 8) | regl);
              regf = (regf & (FLAG3 | FLAG5)) | flagsszp[rega];
              z80cycle = z80cycle + 7;
              break;
            case 0xb7 :         /* OR A */
              regf = (regf & (FLAG3 | FLAG5)) | flagsszp[rega];
              z80cycle = z80cycle + 4;
              break;
            case 0xb8 :         /* CP B */                              // !
              comparea(regb);
              z80cycle = z80cycle + 4;
              break;
            case 0xb9 :         /* CP C */                              // !
              comparea(regc);
              z80cycle = z80cycle + 4;
              break;
            case 0xba :         /* CP D */                              // !
              comparea(regd);
              z80cycle = z80cycle + 4;
              break;
            case 0xbb :         /* CP E */                              // !
              comparea(rege);
              z80cycle = z80cycle + 4;
              break;
            case 0xbc :         /* CP H */                              // !
              comparea(regh);
              z80cycle = z80cycle + 4;
              break;
            case 0xbd :         /* CP L */                              // !
              comparea(regl);
              z80cycle = z80cycle + 4;
              break;
            case 0xbe :         /* CP (HL) */                           // !
              comparea(mread((regh << 8) | regl));
              z80cycle = z80cycle + 7;
              break;
            case 0xbf :         /* CP A */                              // !
              comparea(rega);
              z80cycle = z80cycle + 4;
              break;
            case 0xc0 :         /* RET NZ */                            // !
              if (!(regf & ZFLAG)) {
                regpc = z80pullstack() | (z80pullstack() << 8);
                z80cycle = z80cycle + 11;
              }
              else {
                z80cycle = z80cycle + 5;
              }
              break;
            case 0xc1 :         /* POP BC */                            // !
              regc = z80pullstack();
              regb = z80pullstack();
              z80cycle = z80cycle + 10;
              break;
            case 0xc2 :         /* JP NZ,$nnnn */                       // !
              bytebuffer = mread(regpc);
              z80_incpc;
              address = (mread(regpc) << 8) | bytebuffer;
              z80_incpc;
              if (!(regf & ZFLAG)) {
                regpc = address;
              }
              z80cycle = z80cycle + 10;
              break;
            case 0xc3 :         /* JP $nnnn */                          // !
              bytebuffer = mread(regpc);
              z80_incpc;
              regpc = (mread(regpc) << 8) | bytebuffer;
              z80cycle = z80cycle + 10;
              break;
            case 0xc4 :         /* CALL NZ,$nnnn */                     // !
              bytebuffer = mread(regpc);
              z80_incpc;
              address = (mread(regpc) << 8) | bytebuffer;
              z80_incpc;
              if (!(regf & ZFLAG)) {
                z80pushstack(regpc >> 8);
                z80pushstack(regpc);
                regpc = address;
                z80cycle = z80cycle + 17;
              }
              else {
                z80cycle = z80cycle + 10;
              }
              break;
            case 0xc5 :         /* PUSH BC */                           // !
              z80pushstack(regb);
              z80pushstack(regc);
              z80cycle = z80cycle + 11;
              break;
            case 0xc6 :         /* ADD A,$nn */
              adda(mread(regpc));
              z80_incpc;
              z80cycle = z80cycle + 7;
              break;
            case 0xc7 :         /* RST 0 */
              z80pushstack(regpc >> 8);
              z80pushstack(regpc);
              regpc = (mread(1) << 8) | mread(0);
              z80cycle = z80cycle + 11;
              break;
            case 0xc8 :         /* RET Z */                             // !
              if (regf & ZFLAG) {
                regpc = z80pullstack() | (z80pullstack() << 8);
                z80cycle = z80cycle + 11;
              }
              else {
                z80cycle = z80cycle + 5;
              }
              break;
            case 0xc9 :         /* RET */                               // !
              regpc = z80pullstack() | (z80pullstack() << 8);
              z80cycle = z80cycle + 10;
              break;
            case 0xca :         /* JP Z,$nnnn */                        // !
              bytebuffer = mread(regpc);
              z80_incpc;
              address = (mread(regpc) << 8) | bytebuffer;
              z80_incpc;
              if (regf & ZFLAG) {
                regpc = address;
              }
              z80cycle = z80cycle + 10;
              break;
            case 0xcb : {       /* extra instructions */
              register unsigned char *regptr;
              bytebuffer = mread(regpc);
              z80_incpc;
              switch(bytebuffer) {
                case 0x08 :
                case 0x09 :
                case 0x0a :
                case 0x0b :
                case 0x0c :
                case 0x0d :
                case 0x0f :
                  regptr = regaddr[bytebuffer & 0x7];
//              case 0x0b :     /* RRC E */
                  if (*regptr & 0x01) {
                    *regptr = (*regptr >> 1) | 0x80;
                    regf = flagsszp[*regptr] | CFLAG;
                  }
                  else {
                    *regptr = *regptr >> 1;
                    regf = flagsszp[*regptr];
                  }
                  z80cycle = z80cycle + 8;
                  break;
                case 0x18 :
                case 0x19 :
                case 0x1a :
                case 0x1b :
                case 0x1c :
                case 0x1d :
                case 0x1f :
                  regptr = regaddr[bytebuffer & 0x07];
                  if (regf & CFLAG) {
                    if (*regptr & 0x01) {
                      regf = (regf | CFLAG) & ~(HFLAG | NFLAG);
                    }
                    else {
                      regf = (regf) & ~(CFLAG | HFLAG | NFLAG);
                    }
                    *regptr = (*regptr >> 1) | 0x80;
                  }
                  else {
                    if (*regptr & 0x01) {
                      regf = (regf | CFLAG) & ~(HFLAG | NFLAG);
                    }
                    else {
                      regf = (regf) & ~(CFLAG | HFLAG | NFLAG);
                    }
                    *regptr = *regptr >> 1;
                  }
                  z80cycle = z80cycle + 4;
                  break;
                case 0x40 :     /* BIT 0,reg */
                case 0x41 :
                case 0x42 :
                case 0x43 :
                case 0x44 :
                case 0x45 :
                case 0x47 :
                  regptr = regaddr[bytebuffer & 0x07];
                  if (*regptr & 0x01) {
                    regf = (regf | NFLAG) & ~(ZFLAG | NFLAG);
                  }
                  else {
                    regf = (regf | NFLAG | ZFLAG) & ~(NFLAG);
                  }
                  z80cycle = z80cycle + 8;
                  break;
                case 0x48 :     /* BIT 1,reg */
                case 0x49 :
                case 0x4a :
                case 0x4b :
                case 0x4c :
                case 0x4d :
                case 0x4f :
                  regptr = regaddr[bytebuffer & 0x07];
                  if (*regptr & 0x02) {
                    regf = (regf | NFLAG) & ~(ZFLAG | NFLAG);
                  }
                  else {
                    regf = (regf | NFLAG | ZFLAG) & ~(NFLAG);
                  }
                  z80cycle = z80cycle + 8;
                  break;
                case 0x50 :     /* BIT 2,reg */
                case 0x51 :
                case 0x52 :
                case 0x53 :
                case 0x54 :
                case 0x55 :
                case 0x57 :
                  regptr = regaddr[bytebuffer & 0x07];
                  if (*regptr & 0x04) {
                    regf = (regf | NFLAG) & ~(ZFLAG | NFLAG);
                  }
                  else {
                    regf = (regf | NFLAG | ZFLAG) & ~(NFLAG);
                  }
                  z80cycle = z80cycle + 8;
                  break;
                case 0x58 :     /* BIT 3,reg */
                case 0x59 :
                case 0x5a :
                case 0x5b :
                case 0x5c :
                case 0x5d :
                case 0x5f :
                  regptr = regaddr[bytebuffer & 0x07];
                  if (*regptr & 0x08) {
                    regf = (regf | NFLAG) & ~(ZFLAG | NFLAG);
                  }
                  else {
                    regf = (regf | NFLAG | ZFLAG) & ~(NFLAG);
                  }
                  z80cycle = z80cycle + 8;
                  break;
                case 0x60 :     /* BIT 4,reg */
                case 0x61 :
                case 0x62 :
                case 0x63 :
                case 0x64 :
                case 0x65 :
                case 0x67 :
                  regptr = regaddr[bytebuffer & 0x07];
                  if (*regptr & 0x10) {
                    regf = (regf | NFLAG) & ~(ZFLAG | NFLAG);
                  }
                  else {
                    regf = (regf | NFLAG | ZFLAG) & ~(NFLAG);
                  }
                  z80cycle = z80cycle + 8;
                  break;
                case 0x68 :     /* BIT 5,reg */
                case 0x69 :
                case 0x6a :
                case 0x6b :
                case 0x6c :
                case 0x6d :
                case 0x6f :
                  regptr = regaddr[bytebuffer & 0x07];
                  if (*regptr & 0x20) {
                    regf = (regf | NFLAG) & ~(ZFLAG | NFLAG);
                  }
                  else {
                    regf = (regf | NFLAG | ZFLAG) & ~(NFLAG);
                  }
                  z80cycle = z80cycle + 8;
                  break;
                case 0x70 :     /* BIT 6,reg */
                case 0x71 :
                case 0x72 :
                case 0x73 :
                case 0x74 :
                case 0x75 :
                case 0x77 :
                  regptr = regaddr[bytebuffer & 0x07];
                  if (*regptr & 0x40) {
                    regf = (regf | NFLAG) & ~(ZFLAG | NFLAG);
                  }
                  else {
                    regf = (regf | NFLAG | ZFLAG) & ~(NFLAG);
                  }
                  z80cycle = z80cycle + 8;
                  break;
                case 0x78 :     /* BIT 7,reg */
                case 0x79 :
                case 0x7a :
                case 0x7b :
                case 0x7c :
                case 0x7d :
                case 0x7f :
                  regptr = regaddr[bytebuffer & 0x07];
                  if (*regptr & 0x80) {
                    regf = (regf | NFLAG) & ~(ZFLAG | NFLAG);
                  }
                  else {
                    regf = (regf | NFLAG | ZFLAG) & ~(NFLAG);
                  }
                  z80cycle = z80cycle + 8;
                  break;
                case 0x80 :     /* RES 0,reg */
                case 0x81 :
                case 0x82 :
                case 0x83 :
                case 0x84 :
                case 0x85 :
                case 0x87 :
                  regptr = regaddr[bytebuffer & 0x07];
                  *regptr = *regptr & 0xfe;
                  z80cycle = z80cycle + 8;
                  break;
                case 0x88 :     /* RES 1,reg */
                case 0x89 :
                case 0x8a :
                case 0x8b :
                case 0x8c :
                case 0x8d :
                case 0x8f :
                  regptr = regaddr[bytebuffer & 0x07];
                  *regptr = *regptr & 0xfd;
                  z80cycle = z80cycle + 8;
                  break;
                case 0x90 :     /* RES 2,reg */
                case 0x91 :
                case 0x92 :
                case 0x93 :
                case 0x94 :
                case 0x95 :
                case 0x97 :
                  regptr = regaddr[bytebuffer & 0x07];
                  *regptr = *regptr & 0xfb;
                  z80cycle = z80cycle + 8;
                  break;
                case 0x98 :     /* RES 3,reg */
                case 0x99 :
                case 0x9a :
                case 0x9b :
                case 0x9c :
                case 0x9d :
                case 0x9f :
                  regptr = regaddr[bytebuffer & 0x07];
                  *regptr = *regptr & 0xf7;
                  z80cycle = z80cycle + 8;
                  break;
                case 0xa0 :     /* RES 4,reg */
                case 0xa1 :
                case 0xa2 :
                case 0xa3 :
                case 0xa4 :
                case 0xa5 :
                case 0xa7 :
                  regptr = regaddr[bytebuffer & 0x07];
                  *regptr = *regptr & 0xef;
                  z80cycle = z80cycle + 8;
                  break;
                case 0xa8 :     /* RES 5,reg */
                case 0xa9 :
                case 0xaa :
                case 0xab :
                case 0xac :
                case 0xad :
                case 0xaf :
                  regptr = regaddr[bytebuffer & 0x07];
                  *regptr = *regptr & 0xdf;
                  z80cycle = z80cycle + 8;
                  break;
                case 0xb0 :     /* RES 6,reg */
                case 0xb1 :
                case 0xb2 :
                case 0xb3 :
                case 0xb4 :
                case 0xb5 :
                case 0xb7 :
                  regptr = regaddr[bytebuffer & 0x07];
                  *regptr = *regptr & 0xbf;
                  z80cycle = z80cycle + 8;
                  break;
                case 0xb8 :     /* RES 7,reg */
                case 0xb9 :
                case 0xba :
                case 0xbb :
                case 0xbc :
                case 0xbd :
                case 0xbf :
                  regptr = regaddr[bytebuffer & 0x07];
                  *regptr = *regptr & 0x7f;
                  z80cycle = z80cycle + 8;
                  break;
                default :
                  z80illegal(bytebuffer);
                  return z80stateflags;
              } // switch(bytebuffer) : 0xcb
              break;
              }
            case 0xcc :         /* CALL Z,$nnnn */                      // !
              bytebuffer = mread(regpc);
              z80_incpc;
              address = (mread(regpc) << 8) | bytebuffer;
              z80_incpc;
              if (regf & ZFLAG) {
                z80pushstack(regpc >> 8);
                z80pushstack(regpc);
                regpc = address;
                z80cycle = z80cycle + 17;
              }
              else {
                z80cycle = z80cycle + 10;
              }
              break;
            case 0xcd :         /* CALL $nnnn */                        // !
              bytebuffer = mread(regpc);
              z80_incpc;
              address = (mread(regpc) << 8) | bytebuffer;
              z80_incpc;
              z80pushstack(regpc >> 8);
              z80pushstack(regpc);
              regpc = address;
              z80cycle = z80cycle + 17;
              break;
            case 0xce :         /* ADC A,$nn */
              adca(mread(regpc));
              z80_incpc;
              z80cycle = z80cycle + 7;
              break;
            case 0xcf :         /* RST 8 */
              z80pushstack(regpc >> 8);
              z80pushstack(regpc);
              regpc = (mread(8) << 8) | mread(9);
              z80cycle = z80cycle + 11;
              break;
            case 0xd0 :         /* RET NC */                            // !
              if (!(regf & CFLAG)) {
                regpc = z80pullstack() | (z80pullstack() << 8);
                z80cycle = z80cycle + 11;
              }
              else {
                z80cycle = z80cycle + 5;
              }
              break;
            case 0xd1 :         /* POP DE */                            // !
              rege = z80pullstack();
              regd = z80pullstack();
              z80cycle = z80cycle + 10;
              break;
            case 0xd2 :         /* JP NC,$nnnn */                       // !
              bytebuffer = mread(regpc);
              z80_incpc;
              address = (mread(regpc) << 8) | bytebuffer;
              z80_incpc;
              if (!(regf & CFLAG)) {
                regpc = address;
              }
              z80cycle = z80cycle + 10;
              break;
                                /* OUT ($nn),A still missing */
            case 0xd4 :         /* CALL NC,$nnnn */                     // !
              bytebuffer = mread(regpc);
              z80_incpc;
              address = (mread(regpc) << 8) | bytebuffer;
              z80_incpc;
              if (!(regf & CFLAG)) {
                z80pushstack(regpc >> 8);
                z80pushstack(regpc);
                regpc = address;
                z80cycle = z80cycle + 17;
              }
              else {
                z80cycle = z80cycle + 10;
              }
              break;
            case 0xd5 :         /* PUSH DE */                           // !
              z80pushstack(regd);
              z80pushstack(rege);
              z80cycle = z80cycle + 11;
              break;
            case 0xd6 :         /* SUB $nn */
              suba(mread(regpc));
              z80_incpc;
              z80cycle = z80cycle + 7;
              break;
            case 0xd7 :         /* RST $10 */
              z80pushstack(regpc >> 8);
              z80pushstack(regpc);
              regpc = (mread(0x11) << 8) | mread(0x10);
              z80cycle = z80cycle + 11;
              break;
            case 0xd8 :         /* RET C */                             // !
              if (regf & CFLAG) {
                regpc = z80pullstack() | (z80pullstack() << 8);
                z80cycle = z80cycle + 11;
              }
              else {
                z80cycle = z80cycle + 5;
              }
              break;
            case 0xd9 :         /* EXX */
              regmux     = !regmux;
              bytebuffer = regb;
              regb       = regb2;
              regb2      = bytebuffer;
              bytebuffer = regc;
              regc       = regc2;
              regc2      = bytebuffer;
              bytebuffer = regd;
              regd       = regd2;
              regd2      = bytebuffer;
              bytebuffer = rege;
              rege       = rege2;
              rege2      = bytebuffer;
              bytebuffer = regh;
              regh       = regh2;
              regh2      = bytebuffer;
              bytebuffer = regl;
              regl       = regl2;
              regl2      = bytebuffer;
              z80cycle = z80cycle + 4;
              break;
            case 0xda :         /* JP C,$nnnn */                        // !
              bytebuffer = mread(regpc);
              z80_incpc;
              address = (mread(regpc) << 8) | bytebuffer;
              z80_incpc;
              if (regf & CFLAG) {
                regpc = address;
              }
              z80cycle = z80cycle + 10;
              break;
                                /* IN A,($nn) still missing */
            case 0xdc :         /* CALL C,$nnnn */                      // !
              bytebuffer = mread(regpc);
              z80_incpc;
              address = (mread(regpc) << 8) | bytebuffer;
              z80_incpc;
              if (regf & CFLAG) {
                z80pushstack(regpc >> 8);
                z80pushstack(regpc);
                regpc = address;
                z80cycle = z80cycle + 17;
              }
              else {
                z80cycle = z80cycle + 10;
              }
              break;
            case 0xdd :         /* extra instructions */
              bytebuffer = mread(regpc);
              z80_incpc;
              switch(bytebuffer) {
                case 0x22 :             /* LD ($nnnn),IX */
                  bytebuffer = mread(regpc);
                  z80_incpc;
                  address = (mread(regpc) << 8) | bytebuffer;
                  z80_incpc;
                  mwrite(address, regix & 0xff);
                  mwrite((address + 1) & 0xffff, regix >> 8);
                  z80cycle = z80cycle + 20;
                  break;
                case 0x7c :     /* LD A,HX */
                  rega = regix >> 8;
                  z80cycle = z80cycle + 8;      //?
                  break;
                case 0x7d :     /* LD A,LX */
                  rega = (unsigned char)regix;
                  z80cycle = z80cycle + 8;      //?
                  break;
                case 0xa4 :     /* AND HX */
                  rega = rega & (regix >> 8);
                  regf = (regf & (FLAG3 | FLAG5)) | flagsszp[rega] | HFLAG;
                  z80cycle = z80cycle + 8;      // ?
                  break;
                case 0xa5 :     /* AND LX */
                  rega = rega & regix;
                  regf = (regf & (FLAG3 | FLAG5)) | flagsszp[rega] | HFLAG;
                  z80cycle = z80cycle + 8;      // ?
                  break;
                case 0xbc :     /* CP HX */
                  comparea(regix >> 8);
                  z80cycle = z80cycle + 8;      // ?
                  break;
                case 0xbd :     /* CP LX */
                  comparea(regix);
                  z80cycle = z80cycle + 8;      // ?
                  break;
                case 0xe1:      /* POP IX -uso. */
                  regix = z80pullstack() | (z80pullstack() << 8);
                  z80cycle = z80cycle + 10;
                  break;
                case 0xe5:      /* PUSH IX -uso. */
                  z80pushstack(regix);
                  z80pushstack(regix >> 8);
                  z80cycle = z80cycle + 11;
                  break;
                case 0xe9 :     /* JP (IX) */
                  regpc = regix;
                  z80cycle = z80cycle + 8;
                  break;
                case 0xf9 :     /* LD SP,IX */
                  regsp = regix;
                  z80cycle = z80cycle + 10;
                  break;
                default :
                  z80illegal(bytebuffer);
                  return z80stateflags;
              } // switch(bytebuffer) : 0xdd
              break;
            case 0xde :         /* SBC A,$nn */                         // !
              sbca(mread(regpc));
              z80_incpc;
              z80cycle = z80cycle + 7;
              break;
            case 0xdf :         /* RST $18 */
              z80pushstack(regpc >> 8);
              z80pushstack(regpc);
              regpc = (mread(0x19) << 8) | mread(0x18);
              z80cycle = z80cycle + 11;
              break;
            case 0xe0 :         /* RET PO */                            // !
              if (!(regf & PFLAG)) {
                regpc = z80pullstack() | (z80pullstack() << 8);
                z80cycle = z80cycle + 11;
              }
              else {
                z80cycle = z80cycle + 5;
              }
              break;
            case 0xe1 :         /* POP HL */                            // !
              regl = z80pullstack();
              regh = z80pullstack();
              z80cycle = z80cycle + 10;
              break;
            case 0xe2 :         /* JP PO,$nnnn */                       // !
              bytebuffer = mread(regpc);
              z80_incpc;
              address = (mread(regpc) << 8) | bytebuffer;
              z80_incpc;
              if (!(regf & PFLAG)) {
                regpc = address;
              }
              z80cycle = z80cycle + 10;
              break;
            case 0xe3 :         /* EX (SP),HL */                        // !
              bytebuffer = mread(regsp);
              mwrite(regsp, regl);
              regl = bytebuffer;
              bytebuffer = mread((regsp + 1) & 0xffff);
              mwrite((regsp + 1) & 0xffff, regh);
              regh = bytebuffer;
              z80cycle = z80cycle + 19;
              break;
            case 0xe4 :         /* CALL PO,$nnnn */                     // !
              bytebuffer = mread(regpc);
              z80_incpc;
              address = (mread(regpc) << 8) | bytebuffer;
              z80_incpc;
              if (!(regf & PFLAG)) {
                z80pushstack(regpc >> 8);
                z80pushstack(regpc);
                regpc = address;
                z80cycle = z80cycle + 17;
              }
              else {
                z80cycle = z80cycle + 10;
              }
              break;
            case 0xe5 :         /* PUSH HL */                           // !
              z80pushstack(regh);
              z80pushstack(regl);
              z80cycle = z80cycle + 11;
              break;
            case 0xe6 :         /* AND A,$nn */
              rega = rega & mread(regpc);
              z80_incpc;
              regf = (regf & (FLAG3 | FLAG5)) | flagsszp[rega];
              z80cycle = z80cycle + 7;
              break;
            case 0xe7 :         /* RST $20 */
              z80pushstack(regpc >> 8);
              z80pushstack(regpc);
              regpc = (mread(0x21) << 8) | mread(0x20);
              z80cycle = z80cycle + 11;
              break;
            case 0xe8 :         /* RET PE */                            // !
              if (regf & PFLAG) {
                regpc = z80pullstack() | (z80pullstack() << 8);
                z80cycle = z80cycle + 11;
              }
              else {
                z80cycle = z80cycle + 5;
              }
              break;
            case 0xe9 :         /* JP (HL) */                           // !
              regpc = (regh << 8) | regl;
              z80cycle = z80cycle + 4;
              break;
            case 0xea :         /* JP PE,$nnnn */                       // !
              bytebuffer = mread(regpc);
              z80_incpc;
              address = (mread(regpc) << 8) | bytebuffer;
              z80_incpc;
              if (regf & PFLAG) {
                regpc = address;
              }
              z80cycle = z80cycle + 10;
              break;
            case 0xeb :         /* EX DE,HL */                          // !
              bytebuffer = regd;
              regd       = regh;
              regh       = bytebuffer;
              bytebuffer = rege;
              rege       = regl;
              regl       = bytebuffer;
              z80cycle = z80cycle + 4;
              break;
            case 0xec :         /* CALL PE,$nnnn */                     // !
              bytebuffer = mread(regpc);
              z80_incpc;
              address = (mread(regpc) << 8) | bytebuffer;
              z80_incpc;
              if (regf & PFLAG) {
                z80pushstack(regpc >> 8);
                z80pushstack(regpc);
                regpc = address;
                z80cycle = z80cycle + 17;
              }
              else {
                z80cycle = z80cycle + 10;
              }
              break;
            case 0xed :         /* extra instructions */
              bytebuffer = mread(regpc);
              z80_incpc;
              switch(bytebuffer) {

                                        /* Port I/O: these are NOPs on the Apple ][. */
                case 0x40 :             /* IN B,(C) -uso (Not emulated.) */
                  break;
                case 0x41 :             /* OUT (C),B -uso (Not emulated.) */
                  break;

                case 0x42 : {           /* SBC HL,BC -uso. */
                    register int result;
                    result = ((regh << 8) | regl) - ((regb << 8) | regc) - (regf & 1);
                    regf = flagssz [(result >> 8) & 0xff] | ((result >> 16) & CFLAG) | NFLAG |
                      (( regh ^ regb ^ (result >> 8)) & HFLAG) |
                      (((regh ^ regb) & (regb ^ (result >> 8)) >> 5) & PFLAG);
                    regl = (unsigned char)result;
                    regh = (unsigned char)result >> 8;
                  }

/*              { // Mooched from YAZE.
                  unsigned long int result;
                  unsigned short int in_hl, in_bc, in_af, cbits;

                  in_hl=(regh<<8)+regl;
                  in_bc=(regb<<8)+regc;

                  result=(in_hl-in_bc)-(CFLAG?1:0);

                  cbits = (in_hl ^ in_bc ^ result) >> 8;

                  in_af=(rega<<8)+regf;
                  in_af=(in_af& ~0xff) | ((result >> 8) & 0xa8) |
                      (((result & 0xffff) == 0) << 6) |
                      (((cbits >> 6) ^ (cbits >> 5)) & 4) |
                        (cbits & 0x10) | 2 | ((cbits >> 8) & 1);
                  rega=in_af>>8;
                  regf=in_af&0xFF;
                  break;
                }
*/              case 0x43 :             /* LD ($nnnn),BC */
                  bytebuffer = mread(regpc);
                  z80_incpc;
                  address = (mread(regpc) << 8) | bytebuffer;
                  z80_incpc;
                  mwrite(address, regc);
                  mwrite((address + 1) & 0xffff, regb);
                  z80cycle = z80cycle + 20;
                  break;
                case 0x44 :             // NEG
                  regf = (regf & ~(SFLAG | ZFLAG | PFLAG | HFLAG | CFLAG));
                  if (rega == 0x80) {
                    regf = regf | PFLAG;
                  }
                  if (!rega) {
                    regf = regf | CFLAG;
                  }
                  if (!(rega & 0xf)) {
                    regf = regf | HFLAG;
                  }
                  rega = (0-rega);
                  regf = regf | flagssz[rega] | NFLAG;
                  z80cycle = z80cycle + 8;
                  break;
                case 0x46 :             /* IM 0 */
                  bytebuffer = mread(regpc);
                  z80_incpc;
                  z80cycle = z80cycle + 8;      /* ignored */
                  break;
                case 0x47 :             /* LD I,A */
                  regi = rega;
                  z80cycle = z80cycle + 9;
                  break;
                case 0x4a : {           /* ADC HL,BC -uso. */
                    register unsigned int result;
                    result = ((regh << 8) | regl) + ((regb << 8) | regc) + (regf & CFLAG);
                    regf = flagssz [(result >> 8) & 0xff] | ((result >> 16) & CFLAG) |
                      (( regh ^ regb ^ (result >> 8)) & HFLAG) |
                      (((regh ^ regb ^ 0x80) & (regb ^ (result >>8)) >> 5) & PFLAG);
                    regl = (unsigned char)result;
                    regh = (unsigned char)(result >> 8);
                  }
/*                { // Mooched from YAZE.
                  unsigned long int result;
                  unsigned short int in_hl, in_bc, in_af, cbits;

                  in_hl=(regh<<8)+regl;
                  in_bc=(regb<<8)+regc;

                  result=(in_hl+in_bc)+(CFLAG?1:0);

                  cbits = (in_hl ^ in_bc ^ result) >> 8;

                  in_af=(rega<<8)+regf;
                  in_af=(in_af& ~0xff) | ((result >> 8) & 0xa8) |
                      (((result & 0xffff) == 0) << 6) |
                      (((cbits >> 6) ^ (cbits >> 5)) & 4) |
                        (cbits & 0x10) | 2 | ((cbits >> 8) & 1);
                  rega=in_af>>8;
                  regf=in_af&0xFF;
                  break;
                  }
*/              case 0x4b :             /* LD BC,($nnnn) */
                  bytebuffer = mread(regpc);
                  z80_incpc;
                  address = (mread(regpc) << 8) | bytebuffer;
                  z80_incpc;
                  regc = mread(address);
                  regb = mread((address + 1) & 0xffff);
                  z80cycle = z80cycle + 20;
                  break;
                case 0x52 : {           /* SBC HL,DE -uso. */
                    register int result;
                    result = ((regh << 8) | regl) - ((regd << 8) | rege) - (regf & 1);
                    regf = flagssz [(result >> 8) & 0xff] | ((result >> 16) & CFLAG) | NFLAG |
                      (( regh ^ regd ^ (result >> 8)) & HFLAG) |
                      (((regh ^ regd) & (regd ^ (result >> 8)) >> 5) & PFLAG);
                    regl = (unsigned char)result;
                    regh = (unsigned char)result >> 8;
                  }
/*                { // Mooched from YAZE.
                  unsigned long int result;
                  unsigned short int in_hl, in_de, in_af, cbits;

                  in_hl=(regh<<8)+regl;
                  in_de=(regd<<8)+rege;

                  result=(in_hl-in_de)-(CFLAG?1:0);

                  cbits = (in_hl ^ in_de ^ result) >> 8;

                  in_af=(rega<<8)+regf;
                  in_af=(in_af& ~0xff) | ((result >> 8) & 0xa8) |
                      (((result & 0xffff) == 0) << 6) |
                      (((cbits >> 6) ^ (cbits >> 5)) & 4) |
                        (cbits & 0x10) | 2 | ((cbits >> 8) & 1);
                  rega=in_af>>8;
                  regf=in_af&0xFF;
                  break;
                  }
*/              case 0x53 :             /* LD ($nnnn),DE */
                  bytebuffer = mread(regpc);
                  z80_incpc;
                  address = (mread(regpc) << 8) | bytebuffer;
                  z80_incpc;
                  mwrite(address, rege);
                  mwrite((address + 1) & 0xffff, regd);
                  z80cycle = z80cycle + 20;
                  break;
                case 0x56 :             /* IM 1 */
                  bytebuffer = mread(regpc);
                  z80_incpc;
                  z80cycle = z80cycle + 8;      /* ignored */
                  break;
                case 0x57 :             /* LD A,I */
                  rega = regi;
                  regf = (regf & (FLAG5 | FLAG3 | CFLAG)) | flagssz[rega]
                       | ((regiff2) ? PFLAG : 0);
                  z80cycle = z80cycle + 9;
                  break;
                case 0x5a : {           /* ADC HL,DE -uso. */
                    register unsigned int result;
                    result = ((regh << 8) | regl) + ((regd << 8) | rege) + (regf & CFLAG);
                    regf = flagssz [(result >> 8) & 0xff] | ((result >> 16) & CFLAG) |
                      (( regh ^ regd ^ (result >> 8)) & HFLAG) |
                      (((regh ^ regd ^ 0x80) & (regd ^ (result >>8)) >> 5) & PFLAG);
                    regl = (unsigned char)result;
                    regh = (unsigned char)(result >> 8);
                  }
/*                { // Mooched from YAZE.
                  unsigned long int result;
                  unsigned short int in_hl, in_de, in_af, cbits;

                  in_hl=(regh<<8)+regl;
                  in_de=(regd<<8)+rege;

                  result=(in_hl+in_de)+(CFLAG?1:0);

                  cbits = (in_hl ^ in_de ^ result) >> 8;

                  in_af=(rega<<8)+regf;
                  in_af=(in_af& ~0xff) | ((result >> 8) & 0xa8) |
                      (((result & 0xffff) == 0) << 6) |
                      (((cbits >> 6) ^ (cbits >> 5)) & 4) |
                        (cbits & 0x10) | 2 | ((cbits >> 8) & 1);
                  rega=in_af>>8;
                  regf=in_af&0xFF;
                  break;
                  }
*/              case 0x5b :             /* LD DE,($nnnn) */             //1
                  bytebuffer = mread(regpc);
                  z80_incpc;
                  address = (mread(regpc) << 8) | bytebuffer;
                  z80_incpc;
                  rege = mread(address);
                  regd = mread((address + 1) & 0xffff);
                  z80cycle = z80cycle + 20;
                  break;
                case 0x5e :             /* IM 2 */
                  bytebuffer = mread(regpc);
                  z80_incpc;
                  z80cycle = z80cycle + 8;      /* ignored */
                  break;
                case 0x62 : {           /* SBC HL,HL -uso. */
                    register int result;
                    result = ((regh << 8) | regl) - ((regh << 8) | regl) - (regf & 1);
                    regf = flagssz [(result >> 8) & 0xff] | ((result >> 16) & CFLAG) | NFLAG |
                      (( regh ^ regh ^ (result >> 8)) & HFLAG) |
                      (((regh ^ regh) & (regh ^ (result >> 8)) >> 5) & PFLAG);
                    regl = (unsigned char)result;
                    regh = (unsigned char)result >> 8;
                  }
/*                { // Mooched from YAZE.
                  unsigned long int result;
                  unsigned short int in_hl, in_af, cbits;

                  in_hl=(regh<<8)+regl;

                  result=-(CFLAG?1:0);

                  cbits = result >> 8;

                  in_af=(rega<<8)+regf;
                  in_af=(in_af& ~0xff) | ((result >> 8) & 0xa8) |
                      (((result & 0xffff) == 0) << 6) |
                      (((cbits >> 6) ^ (cbits >> 5)) & 4) |
                        (cbits & 0x10) | 2 | ((cbits >> 8) & 1);
                  rega=in_af>>8;
                  regf=in_af&0xFF;
                  break;
                  }
*/              case 0x63 :             /* LD ($nnnn),HL */             //1
                  bytebuffer = mread(regpc);
                  z80_incpc;
                  address = (mread(regpc) << 8) | bytebuffer;
                  z80_incpc;
                  mwrite(address, regl);
                  mwrite((address + 1) & 0xffff, regh);
                  z80cycle = z80cycle + 20;
                  break;
                case 0x6a : {           /* ADC HL,HL -uso. */
                    register unsigned int result;
                    result = ((regh << 8) | regl) + ((regh << 8) | regl) + (regf & CFLAG);
                    regf = flagssz [(result >> 8) & 0xff] | ((result >> 16) & CFLAG) |
                      (( regh ^ regh ^ (result >> 8)) & HFLAG) |
                      (((regh ^ regh ^ 0x80) & (regh ^ (result >>8)) >> 5) & PFLAG);
                    regl = (unsigned char)result;
                    regh = (unsigned char)(result >> 8);
                  }
/*                { // Mooched from YAZE.
                  unsigned short int in_hl,in_af,result,cbits;
                  in_hl=(regh<<8)+regl;

                  result=(in_hl<<1)+(CFLAG?1:0);

                  cbits = (in_hl ^ in_hl ^ result) >> 8;

                  in_hl=result;
                  in_af=(rega<<8)+regf;
                  in_af=(in_af& ~0xff) | ((result >> 8) & 0xa8) |
                      (((result & 0xffff) == 0) << 6) |
                      (((cbits >> 6) ^ (cbits >> 5)) & 4) |
                        (cbits & 0x10) | 2 | ((cbits >> 8) & 1);
                  rega=in_af>>8;
                  regf=in_af&0xFF;
                  break;
                  }
*/              case 0x6b :             /* LD HL,($nnnn) */             //1
                  bytebuffer = mread(regpc);
                  z80_incpc;
                  address = (mread(regpc) << 8) | bytebuffer;
                  z80_incpc;
                  regl = mread(address);
                  regh = mread((address + 1) & 0xffff);
                  z80cycle = z80cycle + 20;
                  break;
                case 0x72 : {           /* SBC HL,SP -uso. */
                    register int result;
                    result = ((regh << 8) | regl) - regsp - (regf & 1);
                    regf = flagssz [(result >> 8) & 0xff] | ((result >> 16) & CFLAG) | NFLAG |
                      (( regh ^ (regsp >> 8) ^ (result >> 8)) & HFLAG) |
                      (((regh ^ (regsp >> 8)) & ((regsp >> 8) ^ (result >> 8)) >> 5) & PFLAG);
                    regl = (unsigned char)result;
                    regh = (unsigned char)result >> 8;
                  }
/*                { // Mooched from YAZE.
                  unsigned long int result;
                  unsigned short int in_hl, in_af, cbits;

                  in_hl=(regh<<8)+regl;

                  result=(in_hl-regsp)-(CFLAG?1:0);

                  cbits = (in_hl ^ regsp ^ result) >> 8;

                  in_af=(rega<<8)+regf;
                  in_af=(in_af& ~0xff) | ((result >> 8) & 0xa8) |
                      (((result & 0xffff) == 0) << 6) |
                      (((cbits >> 6) ^ (cbits >> 5)) & 4) |
                        (cbits & 0x10) | 2 | ((cbits >> 8) & 1);
                  rega=in_af>>8;
                  regf=in_af&0xFF;
                  break;
                  }
*/              case 0x73 :             /* LD ($nnnn),SP */
                  bytebuffer = mread(regpc);
                  z80_incpc;
                  address = (mread(regpc) << 8) | bytebuffer;
                  z80_incpc;
                  mwrite(address, regsp & 0xff);
                  mwrite((address + 1) & 0xffff, regsp >> 8);
                  z80cycle = z80cycle + 20;
                  break;
                case 0x7a : {           /* ADC HL,SP -uso. */
                    register unsigned int result;
                    result = ((regh << 8) | regl) + regsp + (regf & CFLAG);
                    regf = flagssz [(result >> 8) & 0xff] | ((result >> 16) & CFLAG) |
                      (( regh ^ (regsp >> 8) ^ (result >> 8)) & HFLAG) |
                      (((regh ^ (regsp >> 8) ^ 0x80) & ((regsp >> 8) ^ (result >>8)) >> 5) & PFLAG);
                    regl = (unsigned char)result;
                    regh = (unsigned char)(result >> 8);
                  }
/*                { // Mooched from YAZE.
                  unsigned long int result;
                  unsigned short int in_hl, in_af, cbits;

                  in_hl=(regh<<8)+regl;

                  result=(in_hl+regsp)+(CFLAG?1:0);

                  cbits = (in_hl ^ regsp ^ result) >> 8;

                  in_af=(rega<<8)+regf;
                  in_af=(in_af& ~0xff) | ((result >> 8) & 0xa8) |
                      (((result & 0xffff) == 0) << 6) |
                      (((cbits >> 6) ^ (cbits >> 5)) & 4) |
                        (cbits & 0x10) | 2 | ((cbits >> 8) & 1);
                  rega=in_af>>8;
                  regf=in_af&0xFF;
                  break;
                  }
*/              case 0x7b :             /* LD SP,($nnnn) */
                  bytebuffer = mread(regpc);
                  z80_incpc;
                  address = (mread(regpc) << 8) | bytebuffer;
                  z80_incpc;
                  regsp = mread(address) | (mread((address + 1) & 0xffff) << 8);
                  z80cycle = z80cycle + 20;
                  break;
                case 0xa0 :             /* LDI */
                  mwrite( (regd << 8) | rege, mread( (regh << 8) | regl) );
                  regl++;
                  if (!regl) {
                    regh++;
                  }
                  rege++;
                  if (!rege) {
                    regd++;
                  }
                  if (!regc) {
                    regb--;
                  }
                  regc--;
                  regf = (regf & ~(PFLAG | HFLAG | NFLAG)) | ( (regb | regc) ? PFLAG : 0);
                  z80cycle = z80cycle + 16;
                  break;
                case 0xa1 :             /* CPI */
                  comparea(mread((regh << 8) | regl));
                  regl++;
                  if (!regl) {
                    regh++;
                  }
                  if (!regc) {
                    regb--;
                  }
                  regc--;
                  if (regb | regc) {
                    regf = regf | PFLAG;
                  }
                  else {
                    regf = regf & ~PFLAG;
                  }
                  z80cycle = z80cycle + 16;
                  break;
                case 0xa8 :             /* LDD */
                  mwrite( (regd << 8) | rege, mread( (regh << 8) | regl) );
                  if (!regl) {
                    regh--;
                  }
                  regl--;
                  if (!rege) {
                    regd--;
                  }
                  rege--;
                  if (!regc) {
                    regb--;
                  }
                  regc--;
                  regf = (regf & ~(PFLAG | HFLAG | NFLAG)) | ( (regb | regc) ? PFLAG : 0);
                  z80cycle = z80cycle + 16;
                  break;
                case 0xa9 :             /* CPD */
                  comparea(mread((regh << 8) | regl));
                  if (!regl) {
                    regh--;
                  }
                  regh--;
                  if (!regc) {
                    regb--;
                  }
                  regc--;
                  if (regb | regc) {
                    regf = regf | PFLAG;
                  }
                  else {
                    regf = regf & ~PFLAG;
                  }
                  z80cycle = z80cycle + 16;
                  break;
                case 0xb0 :             /* LDIR */
                  mwrite( (regd << 8) | rege, mread( (regh << 8) | regl) );
                  regl++;
                  if (!regl) {
                    regh++;
                  }
                  rege++;
                  if (!rege) {
                    regd++;
                  }
                  if (!regc) {
                    regb--;
                  }
                  regc--;
                  regf = regf & ~(PFLAG | HFLAG | NFLAG);
                  if (regb | regc) {
                    regpc = (regpc - 2) &0xffff;
                    z80cycle = z80cycle + 21;
                  }
                  else {
                    z80cycle = z80cycle + 16;
                  }
                  break;
                case 0xb1 :             /* CPIR */
                  comparea(mread((regh << 8) | regl));
                  regl++;
                  if (!regl) {
                    regh++;
                  }
                  if (!regc) {
                    regb--;
                  }
                  regc--;
                  if (regb | regc) {
                    regf = regf | PFLAG;
                  }
                  else {
                    regf = regf & ~PFLAG;
                  }
                  if ((regf & PFLAG) && (!(regf & ZFLAG))) {
                    regpc = (regpc - 2) & 0xffff;
                    z80cycle = z80cycle + 21;
                  }
                  else {
                    z80cycle = z80cycle + 16;
                  }
                  break;
                case 0xb8 :             /* LDDR */
                  mwrite( (regd << 8) | rege, mread( (regh << 8) | regl) );
                  if (!regl) {
                    regh--;
                  }
                  regl--;
                  if (!rege) {
                    regd--;
                  }
                  rege--;
                  if (!regc) {
                    regb--;
                  }
                  regc--;
                  regf = regf & ~(PFLAG | HFLAG | NFLAG);
                  if (regb | regc) {
                    regpc = (regpc - 2) &0xffff;
                    z80cycle = z80cycle + 21;
                  }
                  else {
                    z80cycle = z80cycle + 16;
                  }
                  break;
                case 0xb9 :             /* CPDR */
                  comparea(mread((regh << 8) | regl));
                  if (!regl) {
                    regh--;
                  }
                  regh--;
                  if (!regc) {
                    regb--;
                  }
                  regc--;
                  if (regb | regc) {
                    regf = regf | PFLAG;
                  }
                  else {
                    regf = regf & ~PFLAG;
                  }
                  if ((regf & PFLAG) && (!(regf & ZFLAG))) {
                    regpc = (regpc - 2) & 0xffff;
                    z80cycle = z80cycle + 21;
                  }
                  else {
                    z80cycle = z80cycle + 16;
                  }
                  break;
                default :
                  // YAZE: ignore these ops -uso.
                  if (bytebuffer < 0x40 || bytebuffer > 0x7f) {
                    regpc--;
                    break;
                  }
                  z80illegal(bytebuffer);
                  return z80stateflags;
              } // switch(bytebuffer) : 0xed
              break;
            case 0xee :         /* XOR $nn */
              rega = rega ^ mread(regpc);
              z80_incpc;
              regf = (regf & (FLAG3 | FLAG5)) | flagsszp[rega];
              z80cycle = z80cycle + 7;
              break;
            case 0xef :         /* RST $28 */
              z80pushstack(regpc >> 8);
              z80pushstack(regpc);
              regpc = (mread(0x29) << 8) | mread(0x28);
              z80cycle = z80cycle + 11;
              break;
            case 0xf0 :         /* RET P */                             // !
              if (!(regf & SFLAG)) {
                regpc = z80pullstack() | (z80pullstack() << 8);
                z80cycle = z80cycle + 11;
              }
              else {
                z80cycle = z80cycle + 5;
              }
              break;
            case 0xf1 :         /* POP AF */                            // !
              regf = z80pullstack();
              rega = z80pullstack();
              z80cycle = z80cycle + 10;
              break;
            case 0xf2 :         /* JP P,$nnnn */                        // !
              bytebuffer = mread(regpc);
              z80_incpc;
              address = (mread(regpc) << 8) | bytebuffer;
              z80_incpc;
              if (!(regf & SFLAG)) {
                regpc = address;
              }
              z80cycle = z80cycle + 10;
              break;
            case 0xf3 :         /* DI */                                // !
              regiff1 = 0;
              regiff2 = 0;
              z80cycle = z80cycle + 4;
              break;
            case 0xf4 :         /* CALL P,$nnnn */                      // !
              bytebuffer = mread(regpc);
              z80_incpc;
              address = (mread(regpc) << 8) | bytebuffer;
              z80_incpc;
              if (!(regf & SFLAG)) {
                z80pushstack(regpc >> 8);
                z80pushstack(regpc);
                regpc = address;
                z80cycle = z80cycle + 17;
              }
              else {
                z80cycle = z80cycle + 10;
              }
              break;
            case 0xf5 :         /* PUSH AF */                           // !
              z80pushstack(rega);
              z80pushstack(regf);
              z80cycle = z80cycle + 11;
              break;
            case 0xf6 :         /* OR $nn */
              rega = rega | mread(regpc);
              z80_incpc;
              regf = (regf & (FLAG3 | FLAG5)) | flagsszp[rega];
              z80cycle = z80cycle + 7;
              break;
            case 0xf7 :         /* RST $30 */
              z80pushstack(regpc >> 8);
              z80pushstack(regpc);
              regpc = (mread(0x31) << 8) | mread(0x30);
              z80cycle = z80cycle + 11;
              break;
            case 0xf8 :         /* RET M */                             // !
              if (regf & SFLAG) {
                regpc = z80pullstack() | (z80pullstack() << 8);
                z80cycle = z80cycle + 11;
              }
              else {
                z80cycle = z80cycle + 5;
              }
              break;
            case 0xf9 :         /* LD SP,HL */                          // !
              regsp = (regh << 8) | regl;
              z80cycle = z80cycle + 0;  // +?
              break;
            case 0xfa :         /* JP M,$nnnn */                        // !
              bytebuffer = mread(regpc);
              z80_incpc;
              address = (mread(regpc) << 8) | bytebuffer;
              z80_incpc;
              if (regf & SFLAG) {
                regpc = address;
              }
              z80cycle = z80cycle + 10;
              break;
            case 0xfb :         /* EI */                                // !
              regiff2 = 1;
              regiff2 = 1;
              z80cycle = z80cycle + 4;
              break;
            case 0xfc :         /* CALL M,$nnnn */                      // !
              bytebuffer = mread(regpc);
              z80_incpc;
              address = (mread(regpc) << 8) | bytebuffer;
              z80_incpc;
              if (regf & SFLAG) {
                z80pushstack(regpc >> 8);
                z80pushstack(regpc);
                regpc = address;
                z80cycle = z80cycle + 17;
              }
              else {
                z80cycle = z80cycle + 10;
              }
              break;
            case 0xfd :         // extra instructions
              bytebuffer = mread(regpc);
              z80_incpc;
              switch(bytebuffer) {
                case 0x22 :     /* LD ($nnnn),IY */
                  bytebuffer = mread(regpc);
                  z80_incpc;
                  address = (mread(regpc) << 8) | bytebuffer;
                  z80_incpc;
                  mwrite(address, regiy & 0xff);
                  mwrite((address + 1) & 0xffff, regiy >> 8);
                  z80cycle = z80cycle + 20;
                  break;
                case 0x7c :     /* LD A,HY */
                  rega = regiy >> 8;
                  z80cycle = z80cycle + 4;      //?
                  break;
                case 0x7d :     /* LD A,LY */
                  rega = (unsigned char)regiy;
                  z80cycle = z80cycle + 4;      //?
                  break;
                case 0xa4 :     /* AND HY */
                  rega = rega & (regiy >> 8);
                  regf = (regf & (FLAG3 | FLAG5)) | flagsszp[rega] | HFLAG;
                  z80cycle = z80cycle + 8;      // ?
                  break;
                case 0xa5 :     /* AND LY */
                  rega = rega & regiy;
                  regf = (regf & (FLAG3 | FLAG5)) | flagsszp[rega] | HFLAG;
                  z80cycle = z80cycle + 8;      // ?
                  break;
                case 0xbc :     /* CP HY */
                  comparea(regiy >> 8);
                  z80cycle = z80cycle + 8;      // ?
                  break;
                case 0xbd :     /* CP LY */
                  comparea(regiy);
                  z80cycle = z80cycle + 8;      // ?
                  break;
                case 0xe1:      /* POP IY -uso. */
                  regiy = z80pullstack() | (z80pullstack() << 8);
                  z80cycle = z80cycle + 10;
                  break;
                case 0xe5:      /* PUSH IY -uso. */
                  z80pushstack(regiy);
                  z80pushstack(regiy >> 8);
                  z80cycle = z80cycle + 11;
                  break;
                case 0xe9 :     /* JP (IY) */
                  regpc = regiy;
                  z80cycle = z80cycle + 8;
                  break;
                case 0xf9 :     /* LD SP,IY */
                  regsp = regiy;
                  break;
                default :
                  z80illegal(bytebuffer);
                  return z80stateflags;
              } // switch(bytebuffer) : 0xfd
              break;
            case 0xfe :         /* CP $nn */                            // !
              comparea(mread(regpc));
              z80_incpc;
              z80cycle = z80cycle + 7;
              break;
            case 0xff :         /* RST $38 */
              z80pushstack(regpc >> 8);
              z80pushstack(regpc);
              regpc = (mread(0x39) << 8) | mread(0x38);
              z80cycle = z80cycle + 11;
              break;
            default :
              z80illegal(bytebuffer);
              return z80stateflags;

          } // switch(bytebuffer) : main instructions

          {
            register unsigned int delay;
            for (delay=z80delay; delay; delay--);
          }

        } // do
        while (z80traceflag != 1);
        z80stateflags = z80stateflags | CPU_STATETRACE;

        setmessage("Z80 stopped because of trace mode");
        z80light(0);

        return z80stateflags;

    } // z80line


/*--------------------------------------*/


      unsigned char *z80new(slot *slotpointer, unsigned int slotnumber) {

        slotpointer->slotclose          = (void *)&slotnofunction;
        slotpointer->slotreset          = (void *)&z80reset;
        slotpointer->slotstore          = (void *)&z80store;
        slotpointer->slotrestore        = (void *)&z80restore;
        slotpointer->slotget            = (void *)&z80get;
        slotpointer->slotset            = (void *)&z80set;
        slotpointer->slotromget         = (void *)&z80romget;
        slotpointer->slotromset         = (void *)&z80romset;
        slotpointer->slotmenu           = (void *)&slotnofunction;      // menu functions overridden
        slotpointer->slotshift          = (void *)&slotnofunction;
        slotpointer->slotctrl           = (void *)&slotnofunction;
        slotpointer->slotshiftctrl      = (void *)&slotnofunction;
        slotpointer->slottype           = SLOTTYPEZ80;
        slotpointer->slotdata           = NULL;
        slotpointer->slotname           =
"!\
ECPU;\
GCPU;\
";
        if (!z80present) {
          return taskseterror("No Z80 card present");           // no rom present ==> emulation not possible
        }
        return NULL;    // no error

      } // z80new


/*--------------------------------------*/


      unsigned char *z80store(void *slotdata, unsigned int winprotocol, FILE *file, unsigned int percent) {
        unsigned int  reg;
        unsigned char header[32];

        memset(header, 0x00, sizeof(header));
        strcpy(header, "Z80 STATE V0.32");
        fwrite(header,          sizeof(header), 1, file);

        fwrite(z80rom,          sizeof(z80rom), 1, file);

        fwrite(&rega,           sizeof(rega),           1,file);
        fwrite(&regf,           sizeof(regf),           1,file);
        fwrite(&regb,           sizeof(regb),           1,file);
        fwrite(&regc,           sizeof(regc),           1,file);
        fwrite(&regd,           sizeof(regd),           1,file);
        fwrite(&rege,           sizeof(rege),           1,file);
        fwrite(&regf,           sizeof(regf),           1,file);
        fwrite(&regh,           sizeof(regh),           1,file);
        fwrite(&regl,           sizeof(regl),           1,file);
        fwrite(&rega2,          sizeof(rega2),          1,file);
        fwrite(&regf2,          sizeof(regf2),          1,file);
        fwrite(&regb2,          sizeof(regb2),          1,file);
        fwrite(&regc2,          sizeof(regc2),          1,file);
        fwrite(&regd2,          sizeof(regd2),          1,file);
        fwrite(&rege2,          sizeof(rege2),          1,file);
        fwrite(&regh2,          sizeof(regh2),          1,file);
        fwrite(&regl2,          sizeof(regl2),          1,file);
        fwrite(&regix,          sizeof(regix),          1,file);
        fwrite(&regiy,          sizeof(regiy),          1,file);
        fwrite(&regi,           sizeof(regi),           1,file);
        fwrite(&regr,           sizeof(regr),           1,file);
        fwrite(&regpc,          sizeof(regpc),          1,file);
        fwrite(&regsp,          sizeof(regsp),          1,file);
        fwrite(&regiff1,        sizeof(regiff1),        1,file);
        fwrite(&regiff2,        sizeof(regiff2),        1,file);

        reg = z80getstate();
//      taskloghex32(LOG_ALWAYS, reg);
//      tasklogreturn(LOG_ALWAYS);
        fwrite(&reg,            sizeof(reg),            1,file);
        reg = z80cycle;
        fwrite(&reg,            sizeof(reg),            1,file);
        reg = z80lastcycle;
        fwrite(&reg,            sizeof(reg),            1,file);
        reg = z80linecycles;
        fwrite(&reg,            sizeof(reg),            1,file);
//      reg = z80getdelay();
//      fwrite(&reg,            sizeof(reg),            1,file);

        guipercent(winprotocol, percent,
"!\
EZ80 stored.;\
GZ80 gespeichert.;\
");

        return NULL;    // no error

      } // z80store


/*--------------------------------------*/


      unsigned char *z80restore(void *slotdata, unsigned int winprotocol, FILE *file, unsigned int percent) {
        unsigned int  reg;
        unsigned char header[32];

        fread(header,           sizeof(header),         1,file);
        if (strcmp(header, "Z80 STATE V0.32")) {
          stringwrite(winprotocol, "Z80 emulation data not found.\r");
          return taskseterror("Z80 emulation data not found.");
        }

        fread(z80rom,           sizeof(z80rom),         1,file);

        fread(&rega,            sizeof(rega),           1,file);
        fread(&regf,            sizeof(regf),           1,file);
        fread(&regb,            sizeof(regb),           1,file);
        fread(&regc,            sizeof(regc),           1,file);
        fread(&regd,            sizeof(regd),           1,file);
        fread(&rege,            sizeof(rege),           1,file);
        fread(&regf,            sizeof(regf),           1,file);
        fread(&regh,            sizeof(regh),           1,file);
        fread(&regl,            sizeof(regl),           1,file);
        fread(&rega2,           sizeof(rega2),          1,file);
        fread(&regf2,           sizeof(regf2),          1,file);
        fread(&regb2,           sizeof(regb2),          1,file);
        fread(&regc2,           sizeof(regc2),          1,file);
        fread(&regd2,           sizeof(regd2),          1,file);
        fread(&rege2,           sizeof(rege2),          1,file);
        fread(&regh2,           sizeof(regh2),          1,file);
        fread(&regl2,           sizeof(regl2),          1,file);
        fread(&regix,           sizeof(regix),          1,file);
        fread(&regiy,           sizeof(regiy),          1,file);
        fread(&regi,            sizeof(regi),           1,file);
        fread(&regr,            sizeof(regr),           1,file);
        fread(&regpc,           sizeof(regpc),          1,file);
        fread(&regsp,           sizeof(regsp),          1,file);
        fread(&regiff1,         sizeof(regiff1),        1,file);
        fread(&regiff2,         sizeof(regiff2),        1,file);

        fread(&reg,             sizeof(reg),            1,file);
        taskloghex32(LOG_ALWAYS, reg);
        tasklogreturn(LOG_ALWAYS);
        z80clearstate(-1);      // clear all
        z80setstate(reg);
        fread(&reg,             sizeof(reg),            1,file);
        z80cycle = reg;
        fread(&reg,             sizeof(reg),            1,file);
        z80lastcycle = reg;
        fread(&reg,             sizeof(reg),            1,file);
        z80linecycles = reg;
//      fread(&reg,             sizeof(reg),            1,file);
//      z80setdelay(reg);

        if (z80getstate() & CPU_STATEHALT) {
          z80light(0);
        }
        else {
          z80light(1);
          tasklogstring(LOG_ALWAYS, "Haehhh?");
          tasklogreturn(LOG_ALWAYS);
        }

//      if (cpuwriteregsflag) {
//        if (cpuflag) {
//          imagesettextcolor(window, RGBBLACK, RGBWHITE);
//          imagesetcursor(window, 0, 7);
//          z80writeregs(window);
//        }
//      }

        guipercent(winprotocol, percent,
"!\
EZ80 restored.;\
GZ80 geladen.;\
");

        return NULL;    // no error

      } // z80restore


/*--------------------------------------*/


      unsigned char *z80reset(void *slotdata) {

        z80stateflags = z80stateflags | CPU_STATERESET;
        z80light(0);

        return NULL;    // no error

      } // z80reset


/*--------------------------------------*/


      unsigned char z80get(void *slotdata, unsigned int addr) {

        return virtvideobyte;

      } // z80get


/*--------------------------------------*/


      void z80set(void *slotdata, unsigned int addr, unsigned char value) {

        return;         // Nothing happens here

      } // z80set


/*--------------------------------------*/


      unsigned char z80romget(void *slotdata, unsigned int addr) {

        if (z80active) {
          return z80rom[addr & 0xff];   // read from the rom which was separately loaded
        }
        else {
          return 0xff;
        }

      } // z80romget


/*--------------------------------------*/


      void z80romset(void *slotdata, unsigned int addr, unsigned char value) {

        if (z80active) {
          if ((addr & 0xff) == 0) {
            if (!inidebugflag) {
              imagefillbox(window, 0, 0, 639, 7, RGBBLACK);
              if (cpuflag) {
                cpuflag = 0;            // switch to 6502
                cpufastclearhalt();
                z80stateflags = z80stateflags | CPU_STATEHALT;
                z80light(0);
                cpusetcycle    ( z80cycle      >> 1 );
                cpusetlastcycle( z80lastcycle  >> 1 );
//              cpusetlinecycle( z80linecycles >> 1 );
              }
              else {
                cpuflag = 1;            // switch to Z80
                cpufastsethalt();
                z80stateflags   = z80stateflags & ~(CPU_STATEHALT);
                z80light(1);
                z80lastcycle    = cpugetlastcycle() << 1;
                z80cycle        = cpugetcycle()     << 1;
                z80linecycles   = cpugetlinecycle() << 1;
              }
            }
          }
        }

      } // z80romset




// --> #ifndef DEF_INC_Z80_C
#endif
